{*******************************************************************************
  MM Controls
  Copyright(C) 2005 by Brazil Systems
  Descrio: Componente para o Excel
*******************************************************************************}
{$I mmControls.inc}

unit mmExcel;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ComObj, FileCtrl, xlsConsts {$IFDEF MM_D6_UP}, Variants{$ENDIF};

type
  TCellStyle = (csAutomatic, csChecker, csCrissCross, csDown, csGray16, csGray25, csGray50,
    csGray75, csGray8, csGrid, csHorizontal, csLightDown, csLightHorizontal, csLightUp,
    csLightVertical, csNone, csSemiGray75, csSolid, csUp, csVertical);

  TCellColor = (ccNone, ccBlack, ccWhite, ccRed, ccLime, ccBlue, ccYellow, ccFuchsia, ccCyan,
    ccMaroon, ccGreen, ccNavy, ccOlive, ccPurple, ccTeal, ccSilver, ccGray, ccAutumn,
    ccBullet, ccIvory, ccLavender, ccViolet, ccSalmon, ccWinter, ccPallidBlue, ccDarkPetroleum,
    ccPink, ccYellowColor, ccLightGreen, ccVioletColor, ccMaroonColor, ccPetroleum, ccBlueColor,
    ccTurquoise, ccPallidGreen, ccMoney, ccLightYellow, ccIce, ccLightPink, ccLightViolet,
    ccCream, ccSkyColor, ccWaterGreen, ccLemon, ccGold, ccOrange, ccSummer, ccMediumBlue,
    ccGray40, ccDarkBlue, ccSpring, ccBlackGreen, ccGray80, ccDarkRed, ccBulletColor,
    ccIndianBlue, ccGray90);

  TCellAlignment = (caLeft, caCenter, caRight);

  TFontStyle = (fsNormal, fsItalic, fsBold, fsBoldItalic);

  TXOnBeforeReadCell = procedure(Sender: TObject; var ARow, ACol: Word;
    var Continue: Boolean) of object;
  TXOnAfterReadCell = procedure(Sender: TObject; var ARow, ACol: Word;
    AValue: string) of object;
  TXOnBeforeWriteCell = procedure(Sender: TObject; var ARow, ACol: Word;
    var AValue: string; var Continue: Boolean) of object;
  TXOnAfterWriteCell = procedure(Sender: TObject; var ARow, ACol: Word;
    AValue: string) of object;

  TXOnBeforeWriteFormula = procedure(Sender: TObject; var ARow, ACol: Word;
    var AFormula: string; var Continue: Boolean) of object;
  TXOnAfterWriteFormula = procedure(Sender: TObject; var ARow, ACol: Word;
    AValue: string) of object;

  TmmExcel = class(TComponent)
  private
    { Private declarations }
    xlsApp: Variant;
    xlsBook: Variant;
    xlsSheet: Variant;
    xlsSheets: Variant;
    xlsRange: Variant;

    FVisible: Boolean;
    FInitialSheets: Word;
    FCellStyle: TCellStyle;
    FCellColor: TCellColor;
    FCellAlignment: TCellAlignment;
    FFontStyle: TFontStyle;

    FOnAfterReadCell: TXOnAfterReadCell;
    FOnAfterWriteCell: TXOnAfterWriteCell;
    FOnBeforeReadCell: TXOnBeforeReadCell;
    FOnBeforeWriteCell: TXOnBeforeWriteCell;
    FOnAfterWriteFormula: TXOnAfterWriteFormula;
    FOnBeforeWriteFormula: TXOnBeforeWriteFormula;

    procedure Set_Visible(const Value: Boolean);
    procedure Set_InitialSheets(const Value: Word);
    procedure Set_CellStyle(const Value: TCellStyle);
    procedure Set_CellColor(const Value: TCellColor);
    procedure Set_CellAlignment(const Value: TCellAlignment);
    procedure Set_FontStyle(const Value: TFontStyle);

    function Get_Style(ACellStyle: TCellStyle): Integer;
    function Get_Alignment(ACellAlignment: TCellAlignment): Integer;
    function Get_NumberCol(ACol: string): Word;
  protected
    { Protected declarations }
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    function  OpenExcel(AFileName: string=''; ACreate: Boolean=False): Boolean;
    procedure PreviewExcel;
    procedure PrintExcel;
    procedure SaveExcel(AFileName: string='');
    procedure CloseExcel;

    procedure AddSheet;
    procedure ChangeSheet(ASheetNumber: Integer);
    procedure ClearSheet(ASheetNumber: Integer=1);
    procedure DeleteSheet(ASheetNumber: Integer=1);
    procedure HideSheet(ASheetNumber: Integer=1; AVisible: Boolean=True);
    procedure RenameSheet(ASheetName: string; ASheetNumber: Integer=1);

    procedure SelectAll;
    procedure SelectRange(ACellFrom: string='A1'; ACellTo: string='IV65536');

    procedure SetColor(ACellFrom: string='A1'; ACellTo: string='IV65536');
    procedure SetCellColor(ACellColor: TCellColor=ccNone; ACellFrom: string='A1';
      ACellTo: string='IV65536');

    procedure SetStyle(ACellFrom: string='A1'; ACellTo: string='IV65536');
    procedure SetCellStyle(ACellStyle: TCellStyle=csAutomatic; ACellFrom: string='A1';
      ACellTo: string='IV65536');

    procedure SetAlignment(ACellFrom: string='A1'; ACellTo: string='IV65536');
    procedure SetCellAlignment(ACellAlignment: TCellAlignment=caLeft; ACellFrom: string='A1';
      ACellTo: string='IV65536');

    procedure SetCellFull(ACellFrom: string='A1'; ACellTo: string='IV65536';
      ACellColor: TCellColor=ccNone; ACellStyle: TCellStyle=csAutomatic;
      AAlignment: TCellAlignment=caLeft);

    procedure SetColWidth(AColWidth: Word; ACellFrom: string='A1'; ACellTo: string='IV65536');
    procedure SetRowHeight(ARowHeight: Word; ACellFrom: string='A1'; ACellTo: string='IV65536');

    procedure SetCellColWidth(AWidth: Word; ARow, ACol: Word);
    procedure SetCellRowHeight(AHeight: Word; ARow, ACol: Word);

    procedure SetCellFontName(AFontName: string='Arial'; ACellFrom: string='A1';
      ACellTo: string='IV65536');
    procedure SetCellFontColor(AFontColor: TColor=clBlack; ACellFrom: string='A1';
      ACellTo: string='IV65536');
    procedure SetCellFontSize(AFontSize: Word=10; ACellFrom: string='A1'; ACellTo: string='IV65536');
    procedure SetCellFontStyle(AFontStyle: TFontStyle=fsNormal; ACellFrom: string='A1';
      ACellTo: string='IV65536');

    procedure SetCellFontFull(ACellFrom: string='A1'; ACellTo: string='IV65536';
      AFontName: string='Arial'; AFontColor: TColor=clBlack; AFontSize: Word=10;
      AFontStyle: TFontStyle=fsNormal);

    procedure FormulaExcel(ARow, ACol: Word; AFormula: string);
    procedure FormulaExcelEx(ACol: string; ARow: Word; AFormula: string);

    procedure WriteCell(ARow, ACol: Word; AValue: string);
    procedure WriteCellEx(ACol: string; ARow: Word; AValue: string);

    procedure ClearCell(ARow, ACol: Word);
    procedure ClearCellEx(ACol: string; ARow: Word);
    procedure MergeCell(ACellFrom: string='A1'; ACellTo: string='IV65536');

    function ReadCell(ARow, ACol: Word): string;
    function ReadCellEx(ACol: string; ARow: Word): string;

  published
    { Published declarations }
    // Propriedades
    property Visible: Boolean read FVisible write Set_Visible;
    property InitialSheets: Word read FInitialSheets write Set_InitialSheets;
    property CellStyle: TCellStyle read FCellStyle write Set_CellStyle;
    property CellColor: TCellColor read FCellColor write Set_CellColor;
    property CellAlignment: TCellAlignment read FCellAlignment write Set_CellAlignment;
    property FontStyle: TFontStyle read FFontStyle write Set_FontStyle;

    // Eventos
    property OnAfterReadCell: TXOnAfterReadCell read FOnAfterReadCell write FOnAfterReadCell;
    property OnAfterWriteCell: TXOnAfterWriteCell read FOnAfterWriteCell write FOnAfterWriteCell;
    property OnAfterWriteFormula: TXOnAfterWriteFormula read FOnAfterWriteFormula
      write FOnAfterWriteFormula;

    property OnBeforeReadCell: TXOnBeforeReadCell read FOnBeforeReadCell write FOnBeforeReadCell;
    property OnBeforeWriteCell: TXOnBeforeWriteCell read FOnBeforeWriteCell write FOnBeforeWriteCell;
    property OnBeforeWriteFormula: TXOnBeforeWriteFormula read FOnBeforeWriteFormula
      write FOnBeforeWriteFormula;
  end;

implementation

{$R mmExcel.res}

{ TmmExcel }

constructor TmmExcel.Create(AOwner: TComponent);
begin
  FVisible := False;
  FInitialSheets := 1;
  Set_CellStyle(csAutomatic);
  Set_CellColor(ccNone);
  Set_FontStyle(fsNormal);
  inherited Create(AOwner);
end;

destructor TmmExcel.Destroy;
begin
  CloseExcel;
  inherited Destroy;
end;

function TmmExcel.OpenExcel(AFileName: string=''; ACreate: Boolean=False): Boolean;
var
  fileName, dir: string;
begin
  Result := True;
  fileName := Trim(AFileName);

  if fileName <> '' then
  begin
    dir := ExtractFilePath(fileName);
    if dir <> '' then
    begin
      if not DirectoryExists(dir) then
      begin
        if ACreate then
        begin
          if not CreateDir(dir) then
            raise Exception.Create('No foi possvel criar o diretrio ' + dir);
        end
        else
        begin
          Result := False;
          Exit;
        end;
      end;
    end;
    if ExtractFileExt(fileName) = '' then
      fileName := fileName + '.xls';
  end;

  try
    xlsApp := CreateOleObject('Excel.Application');

    if fileName = '' then
    begin
      xlsApp.SheetsInNewWorkbook := FInitialSheets;
      xlsBook := xlsApp.WorkBooks.Add();
      xlsSheet := xlsApp.ActiveSheet;
      xlsSheets := xlsApp.Sheets;
    end
    else
    begin
      if FileExists(fileName) then
      begin
        xlsBook := xlsApp.WorkBooks.Open(fileName);
        xlsSheet := xlsApp.WorkBooks[1].Sheets[1];
        xlsSheets := xlsApp.Sheets;
      end
      else
      begin
        xlsApp.SheetsInNewWorkbook := FInitialSheets;
        xlsBook := xlsApp.WorkBooks.Add();
        xlsSheet := xlsApp.ActiveSheet;
        xlsSheets := xlsApp.Sheets;
        SaveExcel(fileName);
      end;
    end;

    if VarIsEmpty(xlsApp) = True then
    begin
      ShowMessage('No foi possvel abrir o MS Excel');
      Result := False;
      Exit;
    end;
  except
    xlsApp.Quit;
    ShowMessage('No foi possvel abrir o MS Excel');
    Result := False;
  end;
end;

procedure TmmExcel.PreviewExcel;
var
  sheet: Variant;
begin
  sheet := xlsApp.ActiveSheet;
  if VarIsEmpty(sheet) = False then
    sheet.PrintPreview;
end;

procedure TmmExcel.PrintExcel;
var
  sheet: Variant;
begin
  sheet := xlsApp.ActiveSheet;
  if VarIsEmpty(sheet) = False then
    sheet.PrintOut;
end;

procedure TmmExcel.SaveExcel(AFileName: string='');
var
  book: Variant;
begin
  book := xlsApp.ActiveWorkBook;
  if VarIsEmpty(book) = False then
  begin
    if Trim(AFileName) = '' then
      book.Save
    else
      book.SaveAs(AFileName);
  end;
end;

procedure TmmExcel.CloseExcel;
begin
  try
    if not VarIsEmpty(xlsApp) then
    begin
      xlsApp.DisplayAlerts := False;
      xlsApp.Quit;
    end;
  except
    // nada...
  end;
end;

procedure TmmExcel.AddSheet;
begin
  if not VarIsEmpty(xlsBook) then
  begin
    xlsSheet := xlsBook.Sheets.Add(, , 1, xlWorksheet);
    xlsSheets := xlsApp.Sheets;
  end;
end;

procedure TmmExcel.ChangeSheet(ASheetNumber: Integer);
begin
  if VarIsEmpty(xlsApp) = True then Exit;

  if (ASheetNumber = 0) or (ASheetNumber > xlsBook.Sheets.Count) then
    ASheetNumber := 1;

  xlsApp.WorkBooks[1].Sheets[ASheetNumber].Activate;
  xlsSheet := xlsApp.WorkBooks[1].Sheets[ASheetNumber];
end;

procedure TmmExcel.ClearSheet(ASheetNumber: Integer);
begin
  if VarIsEmpty(xlsApp) = True then Exit;
  if ASheetNumber <> 0 then ChangeSheet(ASheetNumber);
  xlsRange := xlsSheet.Range['A1:IV65536'];
  xlsRange.ClearContents;
end;

procedure TmmExcel.DeleteSheet(ASheetNumber: Integer);
var
  sheet, book: Variant;
begin
  if ASheetNumber <> 0 then ChangeSheet(ASheetNumber);
  sheet := xlsApp.ActiveSheet;
  if VarIsEmpty(sheet) = False then
  begin
    book := xlsApp.ActiveWorkBook;
    if book.Sheets.Count > 1 then
      sheet.Delete;
  end;
end;

procedure TmmExcel.HideSheet(ASheetNumber: Integer; AVisible: Boolean);
var
  sheet: Variant;
begin
  if ASheetNumber <> 0 then ChangeSheet(ASheetNumber);
  sheet := xlsApp.ActiveSheet;
  if VarIsEmpty(sheet) = False then
   sheet.Visible := AVisible;
end;

procedure TmmExcel.RenameSheet(ASheetName: string; ASheetNumber: Integer);
var
  sheet: Variant;
begin
  if ASheetNumber <> 0 then ChangeSheet(ASheetNumber);
  sheet := xlsApp.ActiveSheet;
  if VarIsEmpty(sheet) = False then
    sheet.Name := ASheetName;
end;

procedure TmmExcel.SelectAll;
begin
  if VarIsEmpty(xlsApp) = True then Exit;
  xlsSheet.Cells.Select;
  xlsRange := xlsSheet.Range['A1:IV65536'];
end;

procedure TmmExcel.SelectRange(ACellFrom, ACellTo: string);
begin
  if VarIsEmpty(xlsApp) = True then Exit;
  xlsRange := xlsSheet.Range[ACellFrom + ':' + ACellTo];
  xlsRange.Select;
end;

procedure TmmExcel.SetColor(ACellFrom, ACellTo: string);
begin
  if VarIsEmpty(xlsApp) = True then Exit;
  SelectRange(ACellFrom, ACellTo);
  xlsRange.Columns.Interior.ColorIndex := Ord(FCellColor);
end;

procedure TmmExcel.SetCellColor(ACellColor: TCellColor; ACellFrom, ACellTo: string);
begin
  if VarIsEmpty(xlsApp) = True then Exit;
  SelectRange(ACellFrom, ACellTo);
  if Ord(ACellColor) <= 0 then
    xlsRange.Columns.Interior.ColorIndex := Ord(ccWhite)
  else
    xlsRange.Columns.Interior.ColorIndex := Ord(ACellColor);
end;

procedure TmmExcel.SetStyle(ACellFrom, ACellTo: string);
begin
  if VarIsEmpty(xlsApp) = True then Exit;
  SelectRange(ACellFrom, ACellTo);
  xlsRange.Interior.Pattern := Get_Style(FCellStyle);
end;

procedure TmmExcel.SetCellStyle(ACellStyle: TCellStyle; ACellFrom, ACellTo: string);
begin
  if VarIsEmpty(xlsApp) = True then Exit;
  SelectRange(ACellFrom, ACellTo);
  xlsRange.Interior.Pattern := Get_Style(ACellStyle);
end;

procedure TmmExcel.SetAlignment(ACellFrom, ACellTo: string);
begin
  if VarIsEmpty(xlsApp) = True then Exit;
  SelectRange(ACellFrom, ACellTo);
  xlsRange.HorizontalAlignment := Get_Alignment(FCellAlignment);
end;

procedure TmmExcel.SetCellAlignment(ACellAlignment: TCellAlignment;
  ACellFrom, ACellTo: string);
begin
  if VarIsEmpty(xlsApp) = True then Exit;
  SelectRange(ACellFrom, ACellTo);
  xlsRange.HorizontalAlignment := Get_Alignment(ACellAlignment);
end;

procedure TmmExcel.SetCellFull(ACellFrom, ACellTo: string; ACellColor: TCellColor;
  ACellStyle: TCellStyle; AAlignment: TCellAlignment);
begin
  if VarIsEmpty(xlsApp) = True then Exit;
  SelectRange(ACellFrom, ACellTo);
  xlsRange.Columns.Interior.ColorIndex := Ord(ACellColor);
  xlsRange.Interior.Pattern := Get_Style(ACellStyle);
  xlsRange.HorizontalAlignment := Get_Alignment(AAlignment);
end;

procedure TmmExcel.SetColWidth(AColWidth: Word; ACellFrom, ACellTo: string);
begin
  if VarIsEmpty(xlsApp) = True then Exit;
  SelectRange(ACellFrom, ACellTo);
  xlsRange.ColumnWidth := AColWidth;
end;

procedure TmmExcel.SetRowHeight(ARowHeight: Word; ACellFrom, ACellTo: string);
begin
  if VarIsEmpty(xlsApp) = True then Exit;
  SelectRange(ACellFrom, ACellTo);
  xlsRange.RowHeight := ARowHeight;
end;

procedure TmmExcel.SetCellColWidth(AWidth, ARow, ACol: Word);
var
  sheet: Variant;
begin
  sheet := xlsApp.ActiveSheet;
  if VarIsEmpty(sheet) = False then
    sheet.Cells[ARow, ACol].ColumnWidth := AWidth;
end;

procedure TmmExcel.SetCellRowHeight(AHeight, ARow, ACol: Word);
var
  sheet: Variant;
begin
  sheet := xlsApp.ActiveSheet;
  if VarIsEmpty(sheet) = False then
    sheet.Cells[ARow, ACol].RowHeight := AHeight;
end;

procedure TmmExcel.SetCellFontName(AFontName, ACellFrom, ACellTo: string);
begin
  if VarIsEmpty(xlsApp) = True then Exit;
  SelectRange(ACellFrom, ACellTo);
  xlsRange.Font.Name := AFontName;
end;

procedure TmmExcel.SetCellFontColor(AFontColor: TColor; ACellFrom,
  ACellTo: string);
begin
  if VarIsEmpty(xlsApp) = True then Exit;
  SelectRange(ACellFrom, ACellTo);
  xlsRange.Font.Color := AFontColor;
end;

procedure TmmExcel.SetCellFontSize(AFontSize: Word; ACellFrom,
  ACellTo: string);
begin
  if VarIsEmpty(xlsApp) = True then Exit;
  SelectRange(ACellFrom, ACellTo);
  xlsRange.Font.Size := AFontSize;
end;

procedure TmmExcel.SetCellFontStyle(AFontStyle: TFontStyle; ACellFrom,
  ACellTo: string);
begin
  if VarIsEmpty(xlsApp) = True then Exit;
  SelectRange(ACellFrom, ACellTo);

  case AFontStyle of
    fsNormal:
    begin
      xlsRange.Font.Bold := False;
      xlsRange.Font.Italic := False;
    end;
    fsItalic:
    begin
      xlsRange.Font.Bold := False;
      xlsRange.Font.Italic := True;
    end;
    fsBold:
    begin
      xlsRange.Font.Bold := True;
      xlsRange.Font.Italic := False;
    end;
    fsBoldItalic:
    begin
      xlsRange.Font.Bold := True;
      xlsRange.Font.Italic := True;
    end;
  end;
end;

procedure TmmExcel.SetCellFontFull(ACellFrom, ACellTo, AFontName: string;
  AFontColor: TColor; AFontSize: Word; AFontStyle: TFontStyle);
begin
  if VarIsEmpty(xlsApp) = True then Exit;
  SelectRange(ACellFrom, ACellTo);
  xlsRange.Font.Name := AFontName;
  xlsRange.Font.Color := AFontColor;
  xlsRange.Font.Size := AFontSize;
  
  case AFontStyle of
    fsNormal:
    begin
      xlsRange.Font.Bold := False;
      xlsRange.Font.Italic := False;
    end;
    fsItalic:
    begin
      xlsRange.Font.Bold := False;
      xlsRange.Font.Italic := True;
    end;
    fsBold:
    begin
      xlsRange.Font.Bold := True;
      xlsRange.Font.Italic := False;
    end;
    fsBoldItalic:
    begin
      xlsRange.Font.Bold := True;
      xlsRange.Font.Italic := True;
    end;
  end;
end;

procedure TmmExcel.FormulaExcel(ARow, ACol: Word; AFormula: string);
var
  WorkAction: Boolean;
  xValue: string;
begin
  if VarIsEmpty(xlsApp) = True then Exit;

  // Define ao padro
   WorkAction := True;

  // Verifica evento OnBeforeWriteFormula
  if Assigned(FOnBeforeWriteFormula) then
    FOnBeforeWriteFormula(Self, ARow, ACol, AFormula, WorkAction);

  // Verifica se o processo deve continuar
  if (WorkAction = True) then
  begin
    AFormula := Trim(AFormula);
    if AFormula[1] <> '=' then
      AFormula := '=' + AFormula;
    xlsSheet.Cells[ARow, ACol].Formula := AFormula;
    // Verifica evento OnAfterWriteFormula
    if Assigned(FOnAfterWriteFormula) then
    begin
      xValue := ReadCell(ARow, ACol);
      FOnAfterWriteFormula(Self, ARow, ACol, xValue);
    end;
  end;
end;

procedure TmmExcel.FormulaExcelEx(ACol: string; ARow: Word; AFormula: string);
begin
  FormulaExcel(ARow, Get_NumberCol(ACol), AFormula);
end;

procedure TmmExcel.WriteCell(ARow, ACol: Word; AValue: string);
var
  WorkAction: Boolean;
begin
  if VarIsEmpty(xlsApp) = True then Exit;

  // Define ao padro
  WorkAction := True;

  // Verifica evento OnBeforeWriteCell
  if Assigned(FOnBeforeWriteCell) then
    FOnBeforeWriteCell(Self, ARow, ACol, AValue, WorkAction);

  // Verifica se o processo deve continuar
  if (WorkAction = True) then
  begin
    xlsSheet.Cells[ARow, ACol] := AValue;
    // Verifica evento OnAfterWriteCell
    if Assigned(FOnAfterWriteCell) then
      FOnAfterWriteCell(Self, ARow, ACol, AValue);
  end;
end;

procedure TmmExcel.WriteCellEx(ACol: string; ARow: Word; AValue: string);
begin
  WriteCell(ARow, Get_NumberCol(ACol), AValue);
end;

procedure TmmExcel.ClearCell(ARow, ACol: Word);
begin
  if VarIsEmpty(xlsApp) = True then Exit;
  xlsSheet.Cells[ARow, ACol].ClearContents;
end;

procedure TmmExcel.ClearCellEx(ACol: string; ARow: Word);
begin
  ClearCell(ARow, Get_NumberCol(ACol));
end;

function TmmExcel.ReadCell(ARow, ACol: Word): string;
var
  WorkAction: Boolean;
begin
  if VarIsEmpty(xlsApp) = True then Exit;

  // Define ao padro
  WorkAction := True;

  // Verifica evento OnBeforeReadCell
  if Assigned(FOnBeforeReadCell) then
    FOnBeforeReadCell(Self, ARow, ACol, WorkAction);

  // Verifica se o processo deve continuar
  if (WorkAction = True) then
  begin
    Result := xlsSheet.Cells[ARow, ACol].Value;
    // Verifica evento OnAfterReadCell
    if Assigned(FOnAfterReadCell) then
    FOnAfterReadCell(Self, ARow, ACol, Result);
  end;
end;

function TmmExcel.ReadCellEx(ACol: string; ARow: Word): string;
begin
  ReadCell(ARow, Get_NumberCol(ACol));
end;

procedure TmmExcel.Set_Visible(const Value: Boolean);
begin
  if VarIsEmpty(xlsApp) = True then Exit;
  FVisible := Value;
  xlsApp.Visible := FVisible;
end;

procedure TmmExcel.Set_InitialSheets(const Value: Word);
begin
  FInitialSheets := Value;
  if (FInitialSheets < 1) or (FInitialSheets > 255) then
    FInitialSheets := 1;
end;

procedure TmmExcel.Set_CellStyle(const Value: TCellStyle);
begin
  FCellStyle := Value;
end;

procedure TmmExcel.MergeCell(ACellFrom: string='A1'; ACellTo: string='IV65536');
begin
  if VarIsEmpty(xlsApp) = True then Exit;
  SelectRange(ACellFrom, ACellTo);
  xlsRange.Columns.Merge;
end;

procedure TmmExcel.Set_CellColor(const Value: TCellColor);
begin
  FCellColor := Value;
  if Ord(CellColor) = 0 then
    FCellColor := ccWhite;
end;

procedure TmmExcel.Set_FontStyle(const Value: TFontStyle);
begin
  FFontStyle := Value;
end;

function TmmExcel.Get_Style(ACellStyle: TCellStyle): Integer;
begin
  Result := xlPatternAutomatic;
  case ACellStyle of
    csAutomatic:       Result := xlPatternAutomatic;
    csChecker:         Result := xlPatternChecker;
    csCrissCross:      Result := xlPatternCrissCross;
    csDown:            Result := xlPatternDown;
    csGray16:          Result := xlPatternGray16;
    csGray25:          Result := xlPatternGray25;
    csGray50:          Result := xlPatternGray50;
    csGray75:          Result := xlPatternGray75;
    csGray8:           Result := xlPatternGray8;
    csGrid:            Result := xlPatternGrid;
    csHorizontal:      Result := xlPatternHorizontal;
    csLightDown:       Result := xlPatternLightDown;
    csLightHorizontal: Result := xlPatternLightHorizontal;
    csLightUp:         Result := xlPatternLightUp;
    csLightVertical:   Result := xlPatternLightVertical;
    csNone:            Result := xlPatternNone;
    csSemiGray75:      Result := xlPatternSemiGray75;
    csSolid:           Result := xlPatternSolid;
    csUp:              Result := xlPatternUp;
    csVertical:        Result := xlPatternVertical;
  end;
end;

function TmmExcel.Get_NumberCol(ACol: string): Word;
var
  xCol: string;
begin
  xCol := Trim(AnsiUpperCase(ACol));
  if Length(xCol) = 1 then
    Result := Ord(xCol[1]) - 64
  else
    Result := ((Ord(xCol[1]) - 64) * 26) + (Ord(xCol[2]) - 64);
end;

procedure TmmExcel.Set_CellAlignment(const Value: TCellAlignment);
begin
  FCellAlignment := Value;
end;

function TmmExcel.Get_Alignment(ACellAlignment: TCellAlignment): Integer;
begin
  Result := xlLeft;
  case ACellAlignment of
    caLeft:   Result := xlLeft;
    caCenter: Result := xlCenter;
    caRight:  Result := xlRight;
  end;
end;

end.
