unit FileIO;

interface

const CONTROL_VERSION_225 = 20; { Version number for document files which have
                                   been created with the TX4VB 2.25 MDI demo}
  CONTROL_VERSION_230 = 23; { Version number for document files}

type FILE_HEADER = record // Structure for document file header
    lVersion: LongInt;
  end;

procedure FileSaveProc;
procedure FileSaveAsProc;
procedure SaveFile(Filename: string);
procedure OpenFile(Filename: string; FileType: Integer; FileIns: Boolean);
function GetSaveFileName: string;


implementation

uses TXglobal, SysUtils, Dialogs, FEditor;

{-------------------------------------------------------------------------
  FileSaveProc

  Save current text
 -------------------------------------------------------------------------}

procedure FileSaveProc;
var Filename: string;
begin
  with FEditorTexto do begin
    if copy(Caption, 1, 8) = 'Untitled' then begin
       {The file hasn't been saved yet,
        get the filename, then call the save procedure}
      Filename := GetSaveFileName;
    end
    else begin
        // The caption contains the name of the open file
      Filename := Caption;
    end;

    {Call the save procedure, if Filename = Empty then
     the user selected Cancel in the Save As dialog, otherwise
     save the file}
    if Filename <> '' then SaveFile(Filename);
  end;
end;


{-------------------------------------------------------------------------
  FileSaveAsProc

  Get new text filename and saves text
 -------------------------------------------------------------------------}

procedure FileSaveAsProc;
var Filename: string;
begin
  Filename := GetSaveFileName;
  if Filename <> '' then SaveFile(Filename);
end;


{-------------------------------------------------------------------------
  SaveFile

  Save the contents of the active form in the file given in the
  "filename" parameter.

  Parameters: FileName: Name of the file to be loaded (string)
-------------------------------------------------------------------------}

procedure SaveFile(Filename: string);
var FileHeader: FILE_HEADER;
  FileType: Integer;
  lFilePos: LongInt;
  outFile: file of byte;
begin
  FEditorTexto.Cursor := -11; // Hourglass
  with FEditorTexto do begin

    try
     // Determine file type from filename extension
      name := UpperCase(copy(Filename, length(FileName) - 2, 3));
      if name = 'RTF' then
        FileType := RTF_FILE
      else if name = 'TXT' then
        FileType := TXT_FILE
      else if name = 'DOC' then
        FileType := DOC_FILE
      else if name = 'CSS' then
        FileType := CSS_FILE
      else if name = 'HTM' then
        FileType := HTM_FILE
      else if name = 'PDF' then
        FileType := PDF_FILE
      else
        FileType := TXM_FILE;

      case Filetype of
        TXM_FILE: begin // Save RTF file
                  // Open the file
            AssignFile(outFile, filename);
            rewrite(outFile);
                  // Write file header
            FileHeader.lVersion := CONTROL_VERSION_230;
            BlockWrite(outFile, FileHeader, SizeOf(FileHeader));
                  // Write page size and margins
            Doc.lLeftMargin := TX.PageMarginL;
            Doc.lTopMargin := TX.PageMarginT;
            Doc.lRightMargin := TX.PageMarginR;
            Doc.lBottomMargin := TX.PageMarginB;
            Doc.lWidth := TX.PageWidth;
            Doc.lHeight := TX.PageHeight;
            BlockWrite(outFile, Doc, sizeOf(Doc));
            lFilePos := FilePos(outFile);
            CloseFile(outFile);

                  // Write text control contents
            TX.Save(Filename, lFilePos, TXM_FILE, false);
          end
      else begin
          TX.LoadSaveAttribute[0] := TX.PageWidth;
          TX.LoadSaveAttribute[1] := TX.PageHeight;
          TX.LoadSaveAttribute[2] := TX.PageMarginL;
          TX.LoadSaveAttribute[3] := TX.PageMarginT;
          TX.LoadSaveAttribute[4] := TX.PageMarginR;
          TX.LoadSaveAttribute[5] := TX.PageMarginB;
          TX.Save(Filename, 0, FileType, false);
        end;
      end; //Case

     // Set the window caption
      Caption := UpperCase(Filename);
     // reset the dirty flag
      Dirty := False;

    except
      MessageDlg('Error saving file ' + filename, mtError, [mbOK], 0);
      FEditorTexto.Cursor := 0; // default
    end;

    FEditorTexto.Cursor := 0; // default

  end; // with
end;

{-------------------------------------------------------------------------
  GetSaveFileName

  Get a new filename
  Displays a Save As dialog and returns a file name
  or an empty string if the user cancels
 -------------------------------------------------------------------------}

function GetSaveFileName: string;
var Filename: string;
begin
  with FEditorTexto do begin
    SaveFileDialog.Title := 'Save As';
    SaveFileDialog.Filter := 'Word Documento(*.doc)|*.doc';
    SaveFileDialog.DefaultExt := 'txm';
    SaveFileDialog.Filename := '';

    if SaveFileDialog.Execute then
      Filename := SaveFileDialog.Filename
    else Filename := ''; // user canceled dialog

    GetSaveFileName := Filename;
  end;
end;

{-------------------------------------------------------------------------
 OpenFile

 Open the file given in the "filename" parameter, create a new MDI
 child and text control and load the file contents.

 Parameters: FileName: Name of the file to be loaded (string)
             FileType: Type (TXM_FILE ot RTF_FILE)
-------------------------------------------------------------------------}

procedure OpenFile(Filename: string; FileType: Integer; FileIns: Boolean);
var FileHeader: FILE_HEADER;
  lFilePos: LongInt;
  bOpen: boolean;
  InpFile: file of byte;
begin
  bOpen := False;
  FEditorTexto.Cursor := -11; // Hourglass

  with FEditorTexto do begin

    try
      case FileType of
        TXM_FILE: begin
        // Open the selected file
            AssignFile(inpFile, filename);
            reset(inpFile);
            bOpen := True;
        // Read file header
            BlockRead(inpFile, FileHeader, sizeOf(FileHeader));

        {If this is a version 230 file, read page size and margins. Else
         use the defaults from version 2.25}
            if FileHeader.lVersion = CONTROL_VERSION_230 then begin
              BlockRead(inpFile, Doc, sizeOf(Doc));
              TX.PageMarginL := Doc.lLeftMargin;
              TX.PageMarginT := Doc.lTopMargin;
              TX.PageMarginR := Doc.lRightMargin;
              TX.PageMarginB := Doc.lBottomMargin;
              TX.PageWidth := Doc.lWidth;
              TX.PageHeight := Doc.lHeight;
            end
            else begin
              TX.PageMarginL := 1440;
              TX.PageMarginT := 1440;
              TX.PageMarginR := 1440;
              TX.PageMarginB := 1440;
              TX.PageWidth := DINA4_X;
              TX.PageHeight := DINA4_Y;
            end;
            lFilePos := filepos(inpFile);
            CloseFile(inpFile);
            bOpen := False;
            TX.Load(Filename, lFilePos, TXM_FILE, FileIns);
          end
      else begin // Load RTF file
          TX.Load(Filename, 0, FileType, false);
          if TX.LoadSaveAttribute[0] > 0 then
            TX.PageWidth := TX.LoadSaveAttribute[0];
          if TX.LoadSaveAttribute[1] > 0 then
            TX.PageHeight := TX.LoadSaveAttribute[1];
          if TX.LoadSaveAttribute[2] > 0 then
            TX.PageMarginL := TX.LoadSaveAttribute[2];
          if TX.LoadSaveAttribute[3] > 0 then
            TX.PageMarginT := TX.LoadSaveAttribute[3];
          if TX.LoadSaveAttribute[4] > 0 then
            TX.PageMarginR := TX.LoadSaveAttribute[4];
          if TX.LoadSaveAttribute[5] > 0 then
            TX.PageMarginB := TX.LoadSaveAttribute[5];
          TX.SelStart := 0;
        end;
      end; //case

     // Change form's caption and display new text
      Caption := UpperCase(Filename);
      Show;

    except
      MessageDlg('Error opening ' + filename, mtError, [mbOK], 0);
      if bOpen then begin
        closeFile(inpFile);
      end;
      FEditorTexto.Cursor := 0; // default
      close;
    end;

    FEditorTexto.Cursor := 0; //default

  end;
end;

end.
