unit preview;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Forms, Dialogs, Printers,
  StdCtrls, ExtCtrls, Tx4oleLib_TLB, OleCtrls, DbOleCtl, ComCtrls, Controls;

type
  TfrmPrintPreview = class(TForm)
    btnPrint: TButton;
    btnNext: TButton;
    btnPrevious: TButton;
    btnPages: TButton;
    btnZoomIn: TButton;
    btnZoomOut: TButton;
    btnClose: TButton;
    scrollH: TScrollBar;
    scrollV: TScrollBar;
    picMain: TTXTextControl;
    picPage1: TTXTextControl;
    picPage2: TTXTextControl;
    PrintDialog1: TPrintDialog;
    procedure ShowPreview(tx: TTXTextControl);
    procedure FormResize(Sender: TObject);
    procedure SetZoomMode(zoom: Integer);
    procedure ToggleButtons();
    function CalculatePrintZoom(): Boolean;
    procedure DoPreview();
    procedure FormCreate(Sender: TObject);
    procedure btnCloseClick(Sender: TObject);
    procedure btnNextClick(Sender: TObject);
    procedure btnPreviousClick(Sender: TObject);
    procedure btnPagesClick(Sender: TObject);
    procedure btnZoomInClick(Sender: TObject);
    procedure btnZoomOutClick(Sender: TObject);
    procedure scrollVChange(Sender: TObject);
    procedure scrollHChange(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure SubClassProc1(var msg: Tmessage);
    procedure SubClassProc2(var msg: Tmessage);
    procedure StartSubclassing();
    procedure StopSubclassing();
    procedure FormDestroy(Sender: TObject);
    procedure picPage1MouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure btnPrintClick(Sender: TObject);
  end;

var
  frmPrintPreview: TfrmPrintPreview;
  m_TX: TTXTextControl; // Reference to TX that is previewed

  lPageWidth, lPageHeight: LongInt;
  lCurPage: LongInt; // The current page
  bTwoPages: Boolean; // If set to true, two pages will be display in zoom mode 0
  iZoomMode: Integer; // 0 = Total Page, 1 = 75%, 2 = 100%

  iOldPrintZoom: Integer; // Store the old print zoom

  ps1, ps2: tagPAINTSTRUCT;
  DC1, DC2: HDC;
  rect1, rect2: TRect;
  pBrush: HBRUSH;

  hOldProc1: TFarproc;
  hOldProc2: TFarproc;

const BORDER_WIDTH = Int64(8); // Border width around the document
const DBL_BORDER_WIDTH = Int64(18); // Double border width, so it doesn't have to be calculated
const TRI_BORDER_WIDTH = Int64(24); // Triple border width
const crMagnify = 5;


implementation

uses FEditor;

//uses MDIEdit, MDIFrame;

{$R *.DFM}

// ShowPreview
//
// This Method is called to display the PrintPreview dialog
// The TX Text Control that should be previewed has to be
// specified as the parameter

procedure TfrmPrintPreview.ShowPreview(tx: TTXTextControl);
begin
  m_TX := tx;

  lPageWidth := Round(m_TX.PageWidth * (PixelsPerInch / 1440));
  lPageHeight := Round(m_TX.PageHeight * (PixelsPerInch / 1440));
  lCurPage := 1;
  iOldPrintZoom := m_TX.PrintZoom;
  iZoomMode := 0;

  bTwoPages := False;

  picPage2.EditMode := 2;
  picPage2.Left := 2000;
  scrollH.Visible := False;
  scrollV.Visible := False;

  ToggleButtons;

  FormResize(nil);

  frmPrintPreview.ShowModal;
end;

procedure TfrmPrintPreview.FormResize(Sender: TObject);
var
  nWidth, nHeight, nDistX, nDistY: LongInt;
  nMaxScroll: LongInt;
  nInnerHeight: LongInt;
begin
  nInnerHeight := ClientHeight - picMain.Top;

    // Resize the main picture box
  picMain.Width := ClientWidth;
  if (nInnerHeight < 0) then
    picMain.Height := 0
  else
    picMain.Height := nInnerHeight;

    // Adjust the scroll bars
  scrollV.Left := ClientWidth - scrollV.Width;
  if (nInnerHeight - scrollH.Height < 0) then
    scrollV.Height := 0
  else
    scrollV.Height := nInnerHeight - scrollH.Height;

  scrollH.Top := ClientHeight - scrollH.Height;
  if (ClientWidth - scrollV.Width < 0) then
    scrollH.Width := 0
  else
    scrollH.Width := ClientWidth - scrollV.Width;

    // Distinguish between zoom mode 0 and the other two modes here
    // ZoomMode 1 (75%) And 2 (100%)
  if (iZoomMode > 0) then begin
        // Resize the scroll bars and calculate the new ranges
    nMaxScroll := picPage1.Height + DBL_BORDER_WIDTH + scrollH.Height - picMain.Height;

    if (nMaxScroll > 0) then begin
      scrollV.Max := nMaxScroll;
      scrollV.LargeChange := Round(scrollV.Max * (picMain.Height / (picPage1.Height + DBL_BORDER_WIDTH)));
      scrollV.SmallChange := Round(scrollV.Max * 0.2);
    end;

    nMaxScroll := picPage1.Width + DBL_BORDER_WIDTH + scrollV.Width - picMain.Width;

    if (nMaxScroll > 0) then begin
      scrollH.Max := nMaxScroll;
      scrollH.LargeChange := Round(scrollH.Max * (picMain.Width / (picPage1.Width + DBL_BORDER_WIDTH)));
      scrollH.SmallChange := Round(scrollH.Max * 0.2);
    end;

        // Re-adjust the size of the main picture control so the
        // scroll bars are visible
    if (picMain.Width - scrollV.Width > 0) then
      picMain.Width := picMain.Width - scrollV.Width;
    if (picMain.Height - scrollV.Height > 0) then
      picMain.Height := picMain.Height - scrollH.Height;

        // Scroll the picPage to the calculated position
    nDistX := Round((picMain.Width - picPage1.Width) / 2);
    nDistY := Round((picMain.Height - picPage1.Height) / 2);

    if (nDistX < BORDER_WIDTH) then begin
      scrollH.Enabled := True;
      picPage1.Left := -scrollH.Position + BORDER_WIDTH;
    end else begin
      scrollH.Enabled := False;
      picPage1.Left := nDistX;
    end;

    if (nDistY < BORDER_WIDTH) then begin
      scrollV.Enabled := True;
      picPage1.Top := -scrollV.Position + BORDER_WIDTH;
    end else begin
      scrollV.Enabled := False;
      picPage1.Top := nDistY;
    end;

    if (not bTwoPages) then
      picPage2.Left := 2000;

    Exit;
  end;

    // ZoomMode 0 - 1 resp. 2 page Layout - Distiguish between 1 and 2 page layout
  if (bTwoPages) then begin
        // Calculate size of picture boxes
    nWidth := (picMain.Width - (4 * BORDER_WIDTH)) div 2;
    nHeight := lPageHeight * nWidth div lPageWidth;

    if (nHeight > picMain.Height - TRI_BORDER_WIDTH) then begin
      nHeight := picMain.Height - TRI_BORDER_WIDTH;
      nWidth := lPageWidth * nHeight div lPageHeight;

      picPage1.Left := (picMain.Width - 2 * nWidth) div 3;
      picPage1.Top := BORDER_WIDTH;
      picPage2.Left := nWidth + 2 * ((picMain.Width - 2 * nWidth) div 3);
      picPage2.Top := BORDER_WIDTH;
    end else begin
      picPage1.Left := BORDER_WIDTH;
      picPage1.Top := (picMain.Height - nHeight) div 2;
      picPage2.Left := nWidth + DBL_BORDER_WIDTH;
      picPage2.Top := (picMain.Height - nHeight) div 2;
    end;

        // Apply calculated values if they are not negative
    if ((nHeight > 0) and (nWidth > 0)) then begin
      picPage1.Height := nHeight;
      picPage1.Width := nWidth;
      picPage2.Height := nHeight;
      picPage2.Width := nWidth;
    end;
  end else begin
        // Calculate size of txs
    nHeight := picMain.Height - DBL_BORDER_WIDTH;
    nWidth := lPageWidth * nHeight div lPageHeight;

    if (nWidth > picMain.Width - DBL_BORDER_WIDTH) then begin
      nWidth := picMain.Width - DBL_BORDER_WIDTH;
      nHeight := lPageHeight * nWidth div lPageWidth;

      picPage1.Top := (picMain.Height - nHeight) div 2;
      picPage1.Left := BORDER_WIDTH;
    end else begin
      picPage1.Top := BORDER_WIDTH;
      picPage1.Left := (picMain.Width - nWidth) div 2;
    end;

    if ((nHeight > 0) and (nWidth > 0)) then begin
      picPage1.Height := nHeight;
      picPage1.Width := nWidth;
    end;
  end;
end;

// SetZoomMode
//
// Sets the zoom mode specified in the "zoom" parameter. The
// method sets the internal variables to the values required
// for the new mode and hides/shows the scroll bars.

procedure TfrmPrintPreview.SetZoomMode(zoom: Integer);
begin
    // Check for bad zoom values
  if ((zoom < 0) or (zoom > 2)) then Exit;

  iZoomMode := zoom;

  if (iZoomMode = 0) then begin
    scrollH.Visible := False;
    scrollV.Visible := False;
  end else begin
    scrollH.Visible := True;
    scrollV.Visible := True;

    picPage2.Left := 2000;

    if (iZoomMode = 1) then begin
      picPage1.Width := lPageWidth * 3 div 4; // 75%
      picPage1.Height := lPageHeight * 3 div 4;
    end else begin
      picPage1.Width := lPageWidth; // 100%
      picPage1.Height := lPageHeight;
    end;

    if (picPage1.Height + DBL_BORDER_WIDTH - picMain.Height > 0) then
      scrollV.Max := picPage1.Height + DBL_BORDER_WIDTH - picMain.Height
    else
      scrollV.Max := 0;

    if (picPage1.Width + DBL_BORDER_WIDTH - picMain.Width > 0) then
      scrollH.Max := picPage1.Width + DBL_BORDER_WIDTH - picMain.Width
    else
      scrollH.Max := 0;
  end;

  ToggleButtons;

  FormResize(nil);

  DoPreview;
end;

// ToggleButtons
//
// Enables/Disables the buttons according to the current state of the dialog

procedure TfrmPrintPreview.ToggleButtons();
begin
  if (bTwoPages and (iZoomMode = 0)) then
    btnNext.Enabled := (m_TX.CurrentPages > lCurPage + 1)
  else
    btnNext.Enabled := (m_TX.CurrentPages > lCurPage);

  btnPrevious.Enabled := (lCurPage > 1);

  if (iZoomMode > 0) then
    btnPages.Enabled := False
  else
    btnPages.Enabled := (m_TX.CurrentPages > 1);

  btnZoomIn.Enabled := (iZoomMode < 2);
  btnZoomOut.Enabled := (iZoomMode > 0);

  frmPrintPreview.Caption := 'Print Preview - Page ' + IntToStr(lCurPage) + ' of ' + IntToStr(m_TX.CurrentPages);
end;

// CalculatePrintZoom
//
// This function calculates and sets the new print zoom factor
// for the TX. It returns true if successful.

function TfrmPrintPreview.CalculatePrintZoom(): Boolean;
var
  iZoomH, iZoomW: Integer;
begin
  iZoomH := Round(picPage1.Height * 100 / (m_TX.PageHeight * (PixelsPerInch / 1440)));
  iZoomW := Round(picPage1.Width * 100 / (m_TX.PageWidth * (PixelsPerInch / 1440)));

  if iZoomH < iZoomW then begin
    if (iZoomW < 10) then begin
      CalculatePrintZoom := False;
      Exit;
    end;

    m_TX.PrintZoom := iZoomH;
  end else begin
        // 10% is minimum allowed zomm factor
    if (iZoomW < 10) then begin
      CalculatePrintZoom := False;
      Exit;
    end;

    m_TX.PrintZoom := iZoomW;
  end;

  CalculatePrintZoom := True;
end;

// DoPreview
//
// Simply update the picture controls by doing a refresh

procedure TfrmPrintPreview.DoPreview();
begin
  picPage1.Refresh;

  if (bTwoPages) then picPage2.Refresh;
end;


procedure TfrmPrintPreview.FormCreate(Sender: TObject);
var
  style: LongInt;
begin
  style := GetWindowLong(Handle, GWL_STYLE);
  SetWindowLong(Handle, GWL_STYLE, style or WS_CLIPCHILDREN);

  pBrush := INVALID_HANDLE_VALUE;

  StartSubclassing();
end;

procedure TfrmPrintPreview.btnCloseClick(Sender: TObject);
begin
  FEditorTexto.TX.PrintZoom := iOldPrintZoom;
  Close;
end;

procedure TfrmPrintPreview.btnNextClick(Sender: TObject);
begin
  lCurPage := lCurPage + 1;

  ToggleButtons;

  DoPreview;
end;

procedure TfrmPrintPreview.btnPreviousClick(Sender: TObject);
begin
  lCurPage := lCurPage - 1;

  ToggleButtons;

  DoPreview;
end;

procedure TfrmPrintPreview.btnPagesClick(Sender: TObject);
begin
  iZoomMode := 0;

  bTwoPages := not bTwoPages;

  if (bTwoPages) then
    btnPages.Caption := 'One Page'
  else
    btnPages.Caption := 'Two Pages';

  picPage2.Visible := bTwoPages;
  picPage2.Text := '';

  ToggleButtons;

  FormResize(Sender);
end;

procedure TfrmPrintPreview.btnZoomInClick(Sender: TObject);
begin
  SetZoomMode(iZoomMode + 1);
end;

procedure TfrmPrintPreview.btnZoomOutClick(Sender: TObject);
begin
  SetZoomMode(iZoomMode - 1);
end;

procedure TfrmPrintPreview.scrollVChange(Sender: TObject);
begin
  picPage1.Top := -scrollV.Position + BORDER_WIDTH;

  DoPreview;
end;

procedure TfrmPrintPreview.scrollHChange(Sender: TObject);
begin
  picPage1.Left := -scrollH.Position + BORDER_WIDTH;

  DoPreview;
end;

procedure TfrmPrintPreview.FormClose(Sender: TObject;
  var Action: TCloseAction);
begin
  FEditorTexto.TX.PrintZoom := iOldPrintZoom;
  Close;
end;

procedure TfrmPrintPreview.StartSubclassing();
begin
  hOldProc1 := TFarProc(SetWindowLong(picPage1.Handle, GWL_WNDPROC, LongInt(MakeObjectInstance(SubClassProc1))));
  hOldProc2 := TFarProc(SetWindowLong(picPage2.Handle, GWL_WNDPROC, LongInt(MakeObjectInstance(SubClassProc2))));
end;

procedure TfrmPrintPreview.StopSubclassing();
begin
  SetWindowLong(picPage1.Handle, GWL_WNDPROC, longword(hOldProc1));
end;

procedure TfrmPrintPreview.SubClassProc1(var msg: Tmessage);
begin
  if (msg.msg = WM_PAINT) then
  begin
    if (CalculatePrintZoom) then begin
      DC1 := BeginPaint(picPage1.Handle, ps1);
            // GetClientRect would be better here but doesn't seem to work
      GetWindowRect(picPage1.Handle, rect1);
      if (pBrush = INVALID_HANDLE_VALUE) then
        pBrush := GetStockObject(WHITE_BRUSH);
      rect1.Right := rect1.right - rect1.Left;
      rect1.Bottom := rect1.Bottom - rect1.Top;
      rect1.Left := 0;
      rect1.Top := 0;
      Fillrect(DC1, rect1, pBrush);
      m_TX.PrintDevice := DC1;
      m_TX.PrintPage(lCurPage);
      EndPaint(picPage1.Handle, ps1);
      exit;
    end;
  end;
  if (msg.msg = WM_ERASEBKGND) then
    exit;

  msg.result := CallWindowProc(hOldProc1, picPage1.Handle, msg.msg, msg.wparam, msg.lparam);
end;

procedure TfrmPrintPreview.SubClassProc2(var msg: Tmessage);
begin
  if (msg.msg = WM_PAINT) then
  begin
    if (CalculatePrintZoom) then begin
      DC2 := BeginPaint(picPage2.Handle, ps2);
      GetWindowRect(picPage2.Handle, rect2);
      if (pBrush = INVALID_HANDLE_VALUE) then
        pBrush := GetStockObject(WHITE_BRUSH);
      rect2.Right := rect2.right - rect2.Left;
      rect2.Bottom := rect2.Bottom - rect2.Top;
      rect2.Left := 0;
      rect2.Top := 0;
      Fillrect(DC2, rect2, pBrush);
      m_TX.PrintDevice := DC2;
      m_TX.PrintPage(lCurPage + 1);
      EndPaint(picPage2.Handle, ps2);
      exit;
    end;
  end;
  if (msg.msg = WM_ERASEBKGND) then
    exit;

  msg.result := CallWindowProc(hOldProc2, picPage2.Handle, msg.msg, msg.wparam, msg.lparam);
end;

procedure TfrmPrintPreview.FormDestroy(Sender: TObject);
begin
  if (pBrush <> INVALID_HANDLE_VALUE) then
    DeleteObject(pBrush);

  StopSubclassing();
end;

procedure TfrmPrintPreview.picPage1MouseUp(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var
  nOldX, nOldY: LongInt;
begin
  if (Button <> mbLeft) then Exit;

  nOldX := picPage1.Width;
  nOldY := picPage1.Height;

  if (ssCtrl in Shift) then
    SetZoomMode(iZoomMode - 1)
  else begin
    if (iZoomMode = 2) then Exit;
    SetZoomMode(iZoomMode + 1);
  end;


  if (iZoomMode > 0) then begin
    scrollH.Position := Round(scrollH.Max * (X / nOldX));
    scrollV.Position := Round(scrollV.Max * (Y / nOldY));
  end;
end;

procedure TfrmPrintPreview.btnPrintClick(Sender: TObject);
var Copy,
  CurPage,
    StartPage,
    EndPage,
    nTmpPrintZoom: Integer;
begin
  nTmpPrintZoom := m_TX.PrintZoom;
  try
    StartPage := 1;
    EndPage := m_TX.CurrentPages;

  // Initialize and call the common print dialog
    PrintDialog1.Copies := 1;
    PrintDialog1.FromPage := 1;
    PrintDialog1.ToPage := EndPage;
    PrintDialog1.MinPage := 1;
    PrintDialog1.MaxPage := EndPage;
    PrintDialog1.Options := [poPrintToFile, poPageNums];

    if PrintDialog1.Execute then begin
     // Get first and last page
      if poPageNums in PrintDialog1.Options then begin
        StartPage := PrintDialog1.FromPage;
        EndPage := PrintDialog1.ToPage;
      end;

      m_TX.PrintZoom := 100;

     // Print selected pages
      Printer.BeginDoc;
      for Copy := 1 to PrintDialog1.Copies do begin
        for CurPage := StartPage to EndPage do begin
          m_TX.PrintDevice := Printer.CanVas.Handle;
          m_TX.PrintPage(CurPage);
          if (CurPage <> EndPage) then Printer.NewPage;
        end;
      end;
      Printer.EndDoc;
    end;
  except
    on EPrinter do MessageDlg('ERROR printing ' + FEditorTexto.Caption, mtError, [mbOK], 0);
  end;
  m_TX.PrintZoom := nTmpPrintZoom;
end;

end.
