unit UPackage;

interface

uses
  SysUtils, Classes, Contnrs, Windows;

type
  TPackage = class
  private
    // Caminho e nome do arquivo da Package
    FPackageFileName: String;

    // Manipulador da Package
    FPackageHandle: THandle;
    function GetLoaded: Boolean;

  public
    constructor Create(const APackageFileName: String);
    destructor Destroy; override;

    // Carrega a Package para a memria
    procedure LoadPackage;

    // Descarrega a Package da memria
    procedure UnLoadPackage;

    property PackageFileName: String read FPackageFileName;

    // Retorna se a packagee est carregada corretamente
    property Loaded: Boolean read GetLoaded;
  end;

  TPackageClass = class of TPackage;

  TPackages = class(TObjectList)
  private
    FPath: String;
    function  GetItem(Index: Integer): TPackage;
    procedure SetItem(Index: Integer; const Value: TPackage);
    procedure SetPath(const Value: String);
  protected
    FPackageClass: TPackageClass;
  public
    Constructor Create(const APackageClass: TPackageClass; const APath: String);

    procedure AddPackage(const APackageFileName: TFileName);

    // Varre pasta de Packages e carrega uma lista de Objetos TPackage
    procedure LoadPackagesFromDisk;

    property Items[Index: Integer]: TPackage read GetItem write SetItem; default;

    // Pasta onde esto as Packages
    property Path: String read FPath write SetPath;
  end;

var
  RegisteredPackages: TStringList;

implementation

{ TPackage }

constructor TPackage.create(const APackageFileName: String);
begin
  FPackageFileName := APackageFileName;
  FPackageHandle := 0;
end;

destructor TPackage.Destroy;
begin
  if Loaded then
    UnLoadPackage;
  inherited;
end;

function TPackage.GetLoaded: Boolean;
begin
  Result := FPackageHandle <> 0;
end;

procedure TPackage.LoadPackage;
begin
  try
   // Carrega Package
   if FPackageHandle = 0 then
     FPackageHandle := SysUtils.LoadPackage(FPackageFileName);
  except
   FPackageHandle := 0
  end;
end;

procedure TPackage.UnLoadPackage;
begin
  try
   // Descarrega Package
   SysUtils.UnLoadPackage(FPackageHandle);
  except
    // no queremos que levante exceo ao descarregar a package
    // em caso de erro ao descarregar, simplesmente a deixaremos carregada pois
    // ao finalizar a aplicao, ele ser liberada da memoria automaticamente pelo windows
  end;
  FPackageHandle := 0
end;

{ TPackages }

constructor TPackages.Create(const APackageClass: TPackageClass; const APath: String);
begin
  Assert(Assigned(APackageClass));
  inherited Create(true);
  FPackageClass := APackageClass;
  FPath := APath;
end;

function TPackages.GetItem(Index: Integer): TPackage;
begin
    Result := TPackage(inherited GetItem(Index));
end;

procedure TPackages.SetItem(Index: Integer; const Value: TPackage);
begin
 inherited SetItem(Index,Value);
end;

procedure TPackages.SetPath(const Value: String);
begin
  FPath := Value;
  LoadPackagesFromDisk;
end;

procedure TPackages.LoadPackagesFromDisk;
var
  SearchRec: TSearchRec;
begin
  // Limpa Packages se j existirem
  // Lembre-se de que a classe TObjectList j libera
  // os objetos da memria ao limpar
  Clear;
  if FPath <> '' then begin
    FPath := ExcludeTrailingPathDelimiter(FPath)+ '\';
    // Localiza arquivos Package na pasta especificada
    if FindFirst(FPath + '*.bpl',faArchive,SearchRec) = 0 then begin
      AddPackage(FPath + SearchRec.Name);
      while FindNext(SearchRec) = 0 do
        AddPackage(FPath+ SearchRec.Name);
      SysUtils.FindClose(SearchRec)
    end;
  end;
end;

procedure TPackages.AddPackage(const APackageFileName: TFileName);
begin
  // instancia objeto de Package atravs da referencia de classe e adiciona na lista
  Add(FPackageClass.Create(APackageFileName));
end;

initialization
  RegisteredPackages := TStringList.Create;

finalization
 if Assigned(RegisteredPackages) then
   FreeAndNil(RegisteredPackages);

end.
