unit DynamicAppServer;

{$WARN SYMBOL_PLATFORM OFF}

interface

uses
  Classes, DataBkr, Midas, Provider, DynamicDBXModule;

type
  TDynamicAppServer = class(TRemoteDataModule, IAppServer)
  private
    function ObtainConnectionTo(const ProviderName: WideString;
      var IsDynamicModule: boolean): TDynamicModule;
    procedure ReleaseConnectionFrom(DynamicModule: TDynamicModule;
      IsDynamicModule: boolean);
  protected
    function AS_GetProviderNames: OleVariant; safecall;
    function AS_DataRequest(const ProviderName: WideString;
      Data: OleVariant): OleVariant; safecall;
    function AS_ApplyUpdates(const ProviderName: WideString; Delta: OleVariant;
      MaxErrors: Integer; out ErrorCount: Integer;
      var OwnerData: OleVariant): OleVariant; safecall;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    function GetProvider(const ProviderName: string): TCustomProvider; override;
  end;

implementation

uses
  SysUtils, MidConst,
  DynamicProviderList,  DynamicDBXRepositorio;

{$R *.DFM}

{ TDynamicAppServer }

constructor TDynamicAppServer.Create(AOwner: TComponent);
var
  i: integer;
begin
  inherited Create(Aowner);
  if not GetDynProviderNames.IsLoaded then
  begin
    Lock;
    try
      for i := 0 to ComponentCount-1 do
        if Components[i] is TDatasetProvider then
          GetDynProviderNames.Add(Components[i].Name);
    finally
      GetDynProviderNames.IsLoaded := True;
      Unlock;
    end;
  end;
end;

function TDynamicAppServer.GetProvider(
  const ProviderName: string): TCustomProvider;
begin
  Lock;
  try
    try
      Result := inherited GetProvider(ProviderName);
    except
      Result := DynamicRepository.Load(ProviderName, Self);
    end;
    if not Assigned(Result) then
      raise Exception.CreateResFmt(@SProviderNotExported, [ProviderName]);
  finally
    UnLock;
  end;
end;

function TDynamicAppServer.AS_GetProviderNames: OleVariant;
begin
  Lock;
  try
    Result := VarArrayFromStrings(GetDynProviderNames);
  finally
    Unlock;
  end;
end;

function TDynamicAppServer.ObtainConnectionTo(const ProviderName: WideString;
  var IsDynamicModule: boolean): TDynamicModule;
begin
  Result := nil;
  IsDynamicModule := Providers[ProviderName] is TDynamicModule;
  if IsDynamicModule then
  begin
    Result := TDynamicModule(Providers[ProviderName]);
    Result.ObtainConnection;
  end;
end;

procedure TDynamicAppServer.ReleaseConnectionFrom(DynamicModule: TDynamicModule;
  IsDynamicModule: boolean);
begin
  if IsDynamicModule then
    DynamicModule.ReleaseConnection;
end;

function TDynamicAppServer.AS_ApplyUpdates(const ProviderName: WideString;
  Delta: OleVariant; MaxErrors: Integer; out ErrorCount: Integer;
  var OwnerData: OleVariant): OleVariant;
var
  bIsDynamidModule: boolean;
  oDynamicModule: TDynamicModule;
begin
  Lock;
  try
    oDynamicModule := ObtainConnectionTo(ProviderName, bIsDynamidModule);
    Result := inherited AS_ApplyUpdates(ProviderName, Delta, MaxErrors,
      ErrorCount, OwnerData);
    ReleaseConnectionFrom(oDynamicModule, bIsDynamidModule);
  finally
    UnLock;
  end;
end;

function TDynamicAppServer.AS_DataRequest(const ProviderName: WideString;
  Data: OleVariant): OleVariant;
var
  oProvider: TCustomProvider;
begin
  oProvider := Providers[ProviderName];
  if Assigned(oProvider) and (oProvider is TDynamicModule) then
    Result := TDynamicModule(oProvider).CallMethod(Data)
  else
    Result := inherited AS_DataRequest(ProviderName, Data);
end;

end.



