unit DynamicDBXModule;

interface

uses
  Classes, Contnrs, SqlExpr, Provider, DB, DBClient,
  DynamicDBXSequence;

type
  TDynamicModule = class(TDataSetProvider)
  private
    FDataSources: TObjectList;
    FQueries: TObjectList;
  protected
    procedure DoAfterCloseQuery(DataSet: TDataSet); virtual;
    procedure DoBeforeOpenQuery(DataSet: TDataSet); virtual;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure ObtainConnection;
    procedure ReleaseConnection;
    function AddQuery(HasChildren: boolean): TSQLQuery;
    function GetDataSource(const pQueryName: string): TDatasource;
    function GetQueryByName(const pQueryName: string): TSQLQuery;
    function CallMethod(Data: OleVariant): OleVariant; virtual;
  end;

  TDynamicModuleClass = class of TDynamicModule;

  TDynamicModuleAutoInc = class(TDynamicModule)
  private
    FAutoIncRegisters: TStringList;
    FPrimaryKeys: TObjectList;
  protected
    procedure DoBeforeUpdateRecord(SourceDS: TDataSet;
      DeltaDS: TCustomClientDataSet; UpdateKind: TUpdateKind;
      var Applied: Boolean); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure AddPrimaryKey(Kind: TAutoIncKind;
      const QueryName, TableName, FieldName: string;
      const ParentFieldName: String = '';
      const FieldNameToParent: String = '');
  end;

procedure RegisterModule(pClass: TDynamicModuleClass);
procedure RegisterModules(AClasses: array of TDynamicModuleClass);

implementation

uses
  SysUtils, DynamicProviderList, DynamicDBXPool;

procedure RegisterModule(pClass: TDynamicModuleClass);
var
  vProviderName: string;
begin
  Classes.RegisterClass(pClass);
  vProviderName := Copy(pClass.ClassName, 2, Length(pClass.ClassName));
  GetDynProviderNames.Add(vProviderName);
end;

procedure RegisterModules(AClasses: array of TDynamicModuleClass);
var
  I: Integer;
begin
  for I := Low(AClasses) to High(AClasses) do
    RegisterModule(AClasses[I]);
end;
                       
{ TDynamicModule }

constructor TDynamicModule.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FQueries := TObjectList.Create(False);
end;

destructor TDynamicModule.Destroy;
begin
  FQueries.Clear;
  FQueries.Free;
  if Assigned(FDataSources) then
  begin
    FDataSources.Clear;
    FDataSources.Free;
  end;
  inherited Destroy;
end;

function TDynamicModule.AddQuery(HasChildren: boolean): TSQLQuery;
var
  vDataSource: TDataSource;
begin
  Result := TSQLQuery.Create(Self.Owner);
  with Result do
  begin
    BeforeOpen := DoBeforeOpenQuery;
    AfterClose := DoAfterCloseQuery;
  end;
  FQueries.Add(Result);
  if HasChildren then
  begin
    vDataSource := TDatasource.Create(Self.Owner);
    vDataSource.Dataset := Result;
    if not Assigned(FDataSources) then
      FDataSources := TObjectList.Create(False);
    FDataSources.Add(vDataSource);
  end;
end;

function TDynamicModule.GetDataSource(const pQueryName: string): TDatasource;
var
  i: Integer;
begin
  Result := nil;
  if Assigned(FDataSources) then
    for i :=0 to FDataSources.Count-1 do
      if SameText(TDataSource(FDataSources[i]).DataSet.Name, pQueryName) Then
        Result := TDataSource(FDataSources[i]);
end;

function TDynamicModule.GetQueryByName(const pQueryName: string): TSQLQuery;
var
  i: Integer;
begin
  Result := nil;
  for i :=0 to FQueries.Count-1 do
    if CompareText(TSQLQuery(FQueries[i]).Name, pQueryName) = 0 then
    begin
      Result := TSQLQuery(FQueries[i]);
      break;
    end;
end;

procedure TDynamicModule.DoBeforeOpenQuery(DataSet: TDataSet);
begin
  DynamicPool.SetConnection(TSQLQuery(Dataset));
end;

procedure TDynamicModule.DoAfterCloseQuery(DataSet: TDataSet);
begin
  DynamicPool.ReleaseConnection(TSQLQuery(Dataset));
end;

procedure TDynamicModule.ObtainConnection;
var
  i: Integer;
  oConnection: TSQLConnection;
begin
  oConnection := DynamicPool.GetConnection;
  for i := 0 to FQueries.Count-1 do
    TSQLQuery(FQueries[i]).SQLConnection := oConnection;
end;

procedure TDynamicModule.ReleaseConnection;
var
  i: Integer;
begin
  for i := 0 to FQueries.Count-1 do
  begin
    if i = 0 then
      DynamicPool.ReleaseConnection(TSQLQuery(FQueries[i]))
    else
      TSQLQuery(FQueries[i]).SQLConnection := nil;
  end;
end;

function TDynamicModule.CallMethod(Data: OleVariant): OleVariant;
begin
  // Implemented On Child
end;

{ TDynamicModuleAutoInc }

constructor TDynamicModuleAutoInc.Create(AOwner: TComponent);
begin
  inherited;
  FAutoIncRegisters := TStringList.Create;
  FPrimaryKeys := TObjectList.Create(True);
end;

destructor TDynamicModuleAutoInc.Destroy;
begin
  FPrimaryKeys.Free;
  FAutoIncRegisters.Free;
  inherited;
end;

procedure TDynamicModuleAutoInc.AddPrimaryKey(Kind: TAutoIncKind;
  const QueryName, TableName, FieldName: string;
  const ParentFieldName: String = '';
  const FieldNameToParent: String = '');
var
  AutoIncRegister: TPrimaryKey;
begin
  AutoIncRegister := TPrimaryKey.Create;
  AutoIncRegister.TableName := TableName;
  AutoIncRegister.FieldName := FieldName;
  AutoIncRegister.ParentFieldName := ParentFieldName;
  AutoIncRegister.FieldNameToParent := FieldNameToParent;
  AutoIncRegister.Kind := Kind;
  if Kind <> ikMasterTable then
  begin
    if ParentFieldName = EmptyStr then
      AutoIncRegister.ParentFieldName := FieldName;
    if FieldNameToParent = EmptyStr then
      AutoIncRegister.FieldNameToParent := ParentFieldName;
  end;
  FPrimaryKeys.Add(AutoIncRegister);
  FAutoIncRegisters.Add(QueryName);
end;

procedure TDynamicModuleAutoInc.DoBeforeUpdateRecord(SourceDS: TDataSet;
  DeltaDS: TCustomClientDataSet; UpdateKind: TUpdateKind; var Applied: Boolean);
var
  i: integer;
begin
  inherited DoBeforeUpdateRecord(SourceDS, DeltaDS, UpdateKind, Applied);
  if UpdateKind = ukInsert then
  begin
    i := FAutoIncRegisters.IndexOf(SourceDS.Name);
    if i >= 0 then
      DynamicSequence.Execute(SourceDS, DeltaDS, TPrimaryKey(FPrimaryKeys[i]));
  end;
end;

end.

