unit DynamicDBXPool;

interface

uses
  SqlExpr, Classes, SyncObjs;

type
  TDBXConnectionPool = class
  private
    FParams: TStrings;
    FConnectionsInUse: TList;
    FAvailableConnections: TList;
    FMaxConnections: Integer;
    FCS: TCriticalSection;
    FGetDriverFunc: string;
    FLibraryName: string;
    FLoginPrompt: boolean;
    FVendorLib: string;
    FDriverName: string;
    function NewConnection: TSQLConnection;
  public
    constructor Create;
    destructor Destroy; override;
    function GetConnection: TSQLConnection;
    procedure SetConnection(pQuery: TSQLQuery);
    procedure ReturnConnection(Conn: TSQLConnection);
    procedure ReleaseConnection(pQuery: TSQLQuery);
    property MaxConnections: Integer read FMaxConnections write FMaxConnections;
    property Params: TStrings read FParams;
    property LoginPrompt: boolean read FLoginPrompt write FLoginPrompt;
    property DriverName: string read FDriverName write FDriverName;
    property GetDriverFunc: string read FGetDriverFunc write FGetDriverFunc;
    property LibraryName: string read FLibraryName write FLibraryName;
    property VendorLib: string read FVendorLib write FVendorLib;
  end;

function DynamicPool: TDBXConnectionPool;
procedure ConfigPoolToFirebird(pPool: TDBXConnectionPool);

implementation

uses
  SysUtils;

var
  _DBXPool: TDBXConnectionPool;

function DynamicPool: TDBXConnectionPool;
begin
  if not Assigned(_DBXPool) then
    _DBXPool := TDBXConnectionPool.Create;
  Result := _DBXPool;
end;

procedure ConfigPoolToFirebird(pPool: TDBXConnectionPool);
begin
  with pPool do
  begin
    LoginPrompt := False;
    DriverName := 'Interbase';
    GetDriverFunc := 'getSQLDriverINTERBASE';
    LibraryName := 'dbxint30.dll';
    VendorLib := 'GDS32.DLL';
    Params.Values['Database'] := '';
    Params.Values['DriverName'] := 'Interbase';
    Params.Values['User_Name'] := 'sysdba';
    Params.Values['Password'] := 'masterkey';
    Params.Values['SQLDialect'] := '3';
    Params.Values['LocaleCode'] := '0000';
    Params.Values['BlobSize'] := '-1';
    Params.Values['CommitRetain'] := 'False';
    Params.Values['WaitOnLocks'] := 'True';
    Params.Values['Interbase TransIsolation'] := 'ReadCommited';
    Params.Values['Trim Char'] := 'False';
  end;
end;

constructor TDBXConnectionPool.Create;
begin
  inherited;
  FMaxConnections := -1;
  FAvailableConnections := TList.Create;
  FConnectionsInUse := TList.Create;
  FParams := TStringList.Create;
  FCS := TCriticalSection.Create;
end;

destructor TDBXConnectionPool.Destroy;
var
  i: Integer;
begin
  for i := 0 to FConnectionsInUse.Count - 1 do
    TSQLConnection(FConnectionsInUse[i]).Free;
  for i := 0 to FAvailableConnections.Count -1 do
    TSQLConnection(FAvailableConnections[i]).Free;
  FCS.Free;
  FParams.Free;
  FConnectionsInUse.Free;
  FAvailableConnections.Free;
  inherited;
end;

function TDBXConnectionPool.NewConnection: TSQLConnection;
begin
  Result := TSQLConnection.Create(nil);
  with Result do
  begin
    LoginPrompt := FLoginPrompt;
    DriverName:= FDriverName;
    GetDriverFunc := FGetDriverFunc;
    LibraryName := FLibraryName;
    Params.Assign(FParams);
    VendorLib := FVendorLib;
  end;
end;

function TDBXConnectionPool.GetConnection: TSQLConnection;
begin
  Result := nil;
  FCS.Enter;
  try
    if (MaxConnections = 0) then
      raise Exception.Create('No  possvel gerar mais conexes.')
    else if (MaxConnections >= 0)
      and ((FAvailableConnections.Count +
        FConnectionsInUse.Count) >= MaxConnections) then
      raise Exception.Create('No  possvel gerar mais conexes.');
    if FAvailableConnections.Count > 0 then
    begin
      FConnectionsInUse.Add(
        FAvailableConnections[FAvailableConnections.Count-1]);
      FAvailableConnections.Remove(
        FAvailableConnections[FAvailableConnections.Count-1]);
    end
    else
      FConnectionsInUse.Add(NewConnection);
    Result := TSQLConnection(FConnectionsInUse[FConnectionsInUse.Count-1]);
  finally
    FCS.Leave;
  end;
end;

procedure TDBXConnectionPool.SetConnection(
  pQuery: TSQLQuery);
begin
  pQuery.SQLConnection := GetConnection;
end;

procedure TDBXConnectionPool.ReturnConnection(Conn: TSQLConnection);
var
  idx: Integer;
begin
  FCS.Enter;
  try
    idx := FConnectionsInUse.IndexOf(Conn);
    if idx = -1 then
      raise Exception.Create('Conexo no est em uso!');
    FAvailableConnections.Add(FConnectionsInUse[Idx]);
    FConnectionsInUse.Delete(Idx);
  finally
    FCS.Leave;
  end;
end;

procedure TDBXConnectionPool.ReleaseConnection(pQuery: TSQLQuery);
begin
  ReturnConnection(pQuery.SQLConnection);
  pQuery.SQLConnection := nil;
end;

initialization

finalization
  if Assigned(_DBXPool) then
    FreeAndNil(_DBXPool);

end.
