unit DynamicDBXRepositorio;

interface

uses
  SysUtils, Classes, FMTBcd, DB, Provider, SqlExpr,
  DBClient, DynamicDBXModule, WideStrings;

type
  TDBXRepository = class(TDataModule)
    RepConn: TSQLConnection;
    qryQuery: TSQLQuery;
    qryField: TSQLQuery;
    prvQuery: TDataSetProvider;
    prvField: TDataSetProvider;
    qryModule: TSQLQuery;
    prvModule: TDataSetProvider;
    dsModule: TDataSource;
    qryModuleCD_MODULE: TIntegerField;
    qryModuleNM_MODULE: TStringField;
    qryModuleDS_MODULE: TMemoField;
    qryModuleDS_CONFIG: TMemoField;
    qryQueryCD_QUERY: TIntegerField;
    qryQueryNM_QUERY: TStringField;
    qryQueryCD_QUERY_PAI: TIntegerField;
    qryQueryDS_QUERY: TMemoField;
    qryQueryDS_SQL: TMemoField;
    qryQueryDS_CONFIG: TMemoField;
    qryQueryCD_MODULE: TIntegerField;
    dsQuery: TDataSource;
    qryFieldCD_FIELD: TIntegerField;
    qryFieldNM_FIELD: TStringField;
    qryFieldDS_CONFIG: TMemoField;
    qryFieldDS_FIELD: TMemoField;
    qryFieldCD_QUERY: TIntegerField;
    cdsModule: TClientDataSet;
    cdsQuery: TClientDataSet;
    cdsField: TClientDataSet;
    qryParam: TSQLQuery;
    prvParam: TDataSetProvider;
    cdsParam: TClientDataSet;
    qryParamCD_PARAMETER: TIntegerField;
    qryParamNM_PARAMETER: TStringField;
    qryParamTP_DATATYPE: TSmallintField;
    qryParamDS_CONFIG: TMemoField;
    qryParamCD_QUERY: TIntegerField;
    cdsModuleCD_MODULE: TIntegerField;
    cdsModuleNM_MODULE: TStringField;
    cdsModuleDS_MODULE: TMemoField;
    cdsModuleDS_CONFIG: TMemoField;
    cdsQueryCD_QUERY: TIntegerField;
    cdsQueryNM_QUERY: TStringField;
    cdsQueryCD_QUERY_PAI: TIntegerField;
    cdsQueryDS_QUERY: TMemoField;
    cdsQueryDS_SQL: TMemoField;
    cdsQueryDS_CONFIG: TMemoField;
    cdsQueryCD_MODULE: TIntegerField;
    cdsModuleqryQuery: TDataSetField;
    cdsQueryqryParam: TDataSetField;
    cdsQueryqryField: TDataSetField;
    qryModuleCD_QUERY: TIntegerField;
    cdsModuleCD_QUERY: TIntegerField;
    cdsParamCD_PARAMETER: TIntegerField;
    cdsParamNM_PARAMETER: TStringField;
    cdsParamTP_DATATYPE: TSmallintField;
    cdsParamDS_CONFIG: TMemoField;
    cdsParamCD_QUERY: TIntegerField;
    cdsFieldCD_FIELD: TIntegerField;
    cdsFieldNM_FIELD: TStringField;
    cdsFieldDS_CONFIG: TMemoField;
    cdsFieldDS_FIELD: TMemoField;
    cdsFieldCD_QUERY: TIntegerField;
    qryQueryNM_QUERY_PAI: TStringField;
    cdsQueryNM_QUERY_PAI: TStringField;
    qryModuleCD_MODULE_PAI: TIntegerField;
    cdsModuleCD_MODULE_PAI: TIntegerField;
    procedure DataModuleCreate(Sender: TObject);
  private
    { Private declarations }
    procedure ConfigParameters(pQuery: TSQLQuery);
    function QueryHasChildren: boolean;
    function ConfigField(pField: TField): boolean;
    procedure ConfigRelations(pModule: TDynamicModule);
    procedure SetCurrencyField(pField: TField);
    procedure SetDisplayFormat(pField: TField; pFormat: String);
    procedure SetEditFormat(pField: TField; pFormat: String);
    function GetRepConnectionParams: TWideStrings;
    function ConfigQueries(pModule: TDynamicModule): TSQLQuery;
    procedure ConfigFields(pQuery: TSQLQuery);
    function OpenModule(const pModuleName: string): Boolean;
    procedure ConfigModule(pModule: TDynamicModule);
  public
    { Public declarations }
    property ConnectionParams: TWideStrings read GetRepConnectionParams;
    function Load(const pModuleName: string;
      pContainer: TComponent): TCustomProvider;
  end;

function DynamicRepository: TDBXRepository;

implementation

uses
  Variants, DynamicConsts, DynamicUtils, DynamicDBXPool;

var
  _Repositorio: TDBXRepository;

function DynamicRepository: TDBXRepository;
begin
  if not Assigned(_Repositorio) then
    _Repositorio := TDBXRepository.Create(nil);
  Result := _Repositorio;
end;

{$R *.dfm}

{ TDBXRepository }

procedure TDBXRepository.DataModuleCreate(Sender: TObject);
begin
  // *** Repositorio padro. Pegar o caminho de um inifile
  // *** ou setar na aplicao
  RepConn.Params.Values['DATABASE'] :=
    ExtractFilePath(ParamStr(0))+'REPOSITORIO.FDB';
end;

function TDBXRepository.Load(const pModuleName: string;
  pContainer: TComponent{; pConn: TSQLConnection}): TCustomProvider;
var
  vProviderClass: TPersistentClass;
begin
  vProviderClass := Classes.FindClass('T' + pModuleName);
  if not Assigned(vProviderClass) then
    raise Exception.Create(
      Format(cModuleClassNotRegister, ['T' + pModuleName]));
  if not OpenModule(pModuleName) then
    raise Exception.Create(
      Format(cModuleNotFoundOnRepository, [pModuleName]));
  Result := TDynamicModuleClass(vProviderClass).Create(pContainer);
  Result.Name := pModuleName;
  ConfigModule(TDynamicModule(Result));
end;

function TDBXRepository.GetRepConnectionParams: TWideStrings;
begin
  Result := RepConn.Params;
end;

function TDBXRepository.OpenModule(const pModuleName: string): boolean;
begin
  with cdsModule do
  begin
    Close;
    Params[0].AsString := pModuleName;
    // Quando Modulo tem mais de um provider
    Params[1].AsString := pModuleName;
    Open;
    Result := Active and (RecordCount <> 0);
  end;
end;

procedure TDBXRepository.ConfigModule(pModule: TDynamicModule);
begin
  while not cdsModule.Eof do
  begin
    with pModule do
    begin
      Options := [poIncFieldProps, poAutoRefresh,
        poPropogateChanges, poRetainServerOrder];
      UpdateMode := upWhereKeyOnly;
      ConfigQueries(pModule);
      ConfigRelations(pModule);
    end;
    cdsModule.Next;
  end;
end;

function TDBXRepository.ConfigQueries(pModule: TDynamicModule): TSQLQuery;
var
  Query: TSQLQuery;
  Connection: TSQLConnection;
begin
  Result := nil;
  Query := nil;
  Connection := DynamicPool.GetConnection;
  try
    while not cdsQuery.Eof do
    begin
      Query := pModule.AddQuery(QueryHasChildren);
      Query.SQLConnection := Connection;
      with Query do
      begin
        SQL.Text := cdsQueryDS_SQL.AsString;
        Name := cdsQueryNM_QUERY.AsString;
        ConfigParameters(Query);
        ConfigFields(Query);
      end;
      if cdsQueryCD_QUERY.AsInteger =
        cdsModuleCD_QUERY.AsInteger then
        pModule.DataSet := Query;
      cdsQuery.Next;
    end;
  finally
    if Assigned(Query) then
      DynamicPool.ReleaseConnection(Query);
  end;
end;

function TDBXRepository.QueryHasChildren: boolean;
begin
  Result := not VarIsNull(
    cdsQuery.Lookup('CD_QUERY_PAI', cdsQuery['CD_QUERY'], 'CD_QUERY'));
end;

procedure TDBXRepository.ConfigParameters(pQuery: TSQLQuery);
var
  i: integer;
begin
  for i := 0 to pQuery.Params.Count-1 do
    if cdsParam.Locate('NM_PARAMETER', pQuery.Params[I].Name, []) then
    begin
      pQuery.Params[i].ParamType := ptInput;
      pQuery.Params[i].DataType :=
        TFieldType(cdsParamTP_DATATYPE.AsInteger);
    end else
      raise Exception.Create(Format(cParamNotFoundOnRepository,
        [pQuery.Name, pQuery.Params[I].Name]));
end;

procedure TDBXRepository.ConfigFields(pQuery: TSQLQuery);
var
  i: Integer;
  Field: TField;
begin
  pQuery.FieldDefs.Update;
  for i := 0 to pQuery.FieldDefList.Count - 1 do
    with pQuery.FieldDefList[i] do
    begin
      Field := DoCreateField(Self, pQuery, pQuery.FieldDefList[i].Name, '');
      if not ConfigField(Field) then
        raise Exception.Create(Format(cFieldNotFoundOnRepository,
          [pQuery.Name, Field.FieldName]));
    end;
end;

function TDBXRepository.ConfigField(pField: TField): boolean;
var
  sConfig: TStringList;
  i: Integer;
begin
  Result := cdsField.Locate('NM_FIELD', pField.FieldName, []);
  if Result and not cdsFieldDS_CONFIG.IsNull then
  begin
    sConfig := TStringList.Create;
    try
      sConfig.Text := cdsFieldDS_CONFIG.AsString;
      for i := 0 to sConfig.Count - 1 do
      begin
        if SameText(sConfig.Strings[i], 'Required') then
          pField.Required := True
        else if SameText(sConfig.Strings[i], 'pfInKey') then
          pField.ProviderFlags := pField.ProviderFlags + [pfInKey]
        else if SameText(sConfig.Strings[i], 'pfInWhere') then
          pField.ProviderFlags := pField.ProviderFlags + [pfInWhere]
        else if SameText(sConfig.Strings[i], 'pfInUpdate') then
          pField.ProviderFlags := pField.ProviderFlags + [pfInUpdate]
        else if pos('EditMask', sConfig.Strings[i]) <> 0 then
          pField.EditMask := sConfig.ValueFromIndex[i]
        else if SameText(sConfig.Strings[i], 'Currency') then
          SetCurrencyField(pField)
        else if Pos('DisplayFormat', sConfig.Strings[i]) <> 0 then
          SetDisplayFormat(pField, sConfig.ValueFromIndex[i])
        else if Pos('EditFormat', sConfig.Strings[i]) <> 0 then
          SetEditFormat(pField, sConfig.ValueFromIndex[i])
        else if Pos('DefaultExpression', sConfig.Strings[i]) <> 0 then
          pField.DefaultExpression := sConfig.ValueFromIndex[i]
        else if Pos('DisplayLabel', sConfig.Strings[i]) <> 0 then
          pField.DisplayLabel := sConfig.ValueFromIndex[i]
        else if Pos('DisplayWidth', sConfig.Strings[i]) <> 0 then
          pField.DisplayWidth := StrToInt(sConfig.ValueFromIndex[i]);
      end;
    finally
      sConfig.Free;
    end;
  end;
end;

procedure TDBXRepository.SetCurrencyField(pField: TField);
begin
  if pField.DataType = ftFMTBcd then
    TFMTBcdField(pField).Currency := True
  else if pField.DataType = ftFloat then
    TFloatField(pField).Currency := True
  else
    raise Exception.Create(Format(cCannotSetFieldAsCurrency,
      [pField.DataSet.Name, pField.FieldName]));
end;

procedure TDBXRepository.SetDisplayFormat(pField: TField; pFormat: String);
begin
  if pField.DataType in [ftFloat, ftCurrency, ftBCD, ftDate, ftTime, ftDateTime,
    ftTimeStamp, ftFMTBcd] then
    TFloatField(pField).DisplayFormat := pFormat
  else
    raise Exception.Create(Format(cCannotSetDisplayFormatToField,
      [pField.DataSet.Name, pField.FieldName]));
end;

procedure TDBXRepository.SetEditFormat(pField: TField; pFormat: String);
begin
  if pField.DataType in [ftFloat, ftCurrency, ftBCD, ftFMTBcd] then
    TFloatField(pField).EditFormat := pFormat
  else
    raise Exception.Create(Format(cCannotSetEditFormatToField,
      [pField.DataSet.Name, pField.FieldName]));
end;

procedure TDBXRepository.ConfigRelations(pModule: TDynamicModule);
var
  QueryFilho: TSQLQuery;
  DataSourcePai: TDataSource;
begin
  cdsQuery.First;
  while not cdsQuery.Eof do
  begin
    if not cdsQueryNM_QUERY_PAI.IsNull then
    begin
      QueryFilho := pModule.GetQueryByName(cdsQueryNM_QUERY.AsString);
      DataSourcePai := pModule.GetDataSource(cdsQueryNM_QUERY_PAI.AsString);
      if Assigned(DataSourcePai) then
        QueryFilho.DataSource := DataSourcePai;
    end;
    cdsQuery.Next;
  end;
end;

initialization

finalization
  if Assigned(_Repositorio) then
    FreeAndNil(_Repositorio);

end.

