unit DynamicDBXSequence;

interface

uses
  Windows, Classes, DBClient, SqlExpr, DB;

type
  // ikMasterTable    = A chave da tabela  incrementada;
  // ikDetailTable    = A chave da tabela Master e tabelas filhas so
  //      incrementadas s que as filhas possuem campo que apontam para a
  //               master e vo receber o valor dela;
  // ikInheritedTable = A chave da tabela Master e tabelas filhas so
  //      incrementadas com mesmo valor;
  TAutoIncKind = (ikMasterTable, ikDetailTable, ikInheritedTable, ikCustom);

  TGeneratePrimaryKey = procedure(Sender: TObject; const QueryName: string;
    DeltaDS: TCustomClientDataSet) of object;

  TPrimaryKey = class
  private
    FKind: TAutoIncKind;
    FTableName: string;
    FFieldName: string;
    FFieldNameToParent: string;
    FParentFieldName: string;
  public
    property Kind: TAutoIncKind read FKind write FKind;
    property TableName: string read FTableName write FTableName;
    property FieldName: string read FFieldName write FFieldName;
    property FieldNameToParent: string read FFieldNameToParent
      write FFieldNameToParent;
    property ParentFieldName: string read FParentFieldName
      write FParentFieldName;
  end;

  TDynamicSequence = class
  private
    FLock: TRTLCriticalSection;
    procedure Lock;
    procedure UnLock;
    function GetNext(Query: TSQLQuery;
      const sTabela, sCampo: String): Integer;
  public
    constructor Create;
    destructor Destroy; override;
    procedure Execute(SourceDS: TDataSet; DeltaDS: TCustomClientDataSet;
      Pk: TPrimaryKey);
  end;

function DynamicSequence: TDynamicSequence;

implementation

uses SysUtils;

var
  _DynamicSequence: TDynamicSequence;

function DynamicSequence: TDynamicSequence;
begin
  if not Assigned(_DynamicSequence) then
    _DynamicSequence := TDynamicSequence.Create;
  Result := _DynamicSequence;
end;

{ TDynamicSequence }

constructor TDynamicSequence.Create;
begin
  InitializeCriticalSection(FLock);
  inherited Create;
end;

destructor TDynamicSequence.Destroy;
begin
  inherited;
  DeleteCriticalSection(FLock);
end;

procedure TDynamicSequence.Lock;
begin
  EnterCriticalSection(FLock);
end;

procedure TDynamicSequence.UnLock;
begin
  LeaveCriticalSection(FLock);
end;

procedure TDynamicSequence.Execute(SourceDS: TDataSet; DeltaDS:
  TCustomClientDataSet; Pk: TPrimaryKey);
var
  FQuery: TSQLQuery;
begin
  Lock;
  try
    FQuery := TSQLQuery.Create(nil);
    FQuery.SQLConnection := TSQLQuery(SourceDS).SQLConnection;
    case pk.Kind of
      ikMasterTable:
        DeltaDS.FieldByName(pk.FieldName).NewValue :=
          GetNext(FQuery, pk.TableName, pk.FieldName);
      ikDetailTable:
        begin
          DeltaDS.FieldByName(pk.FieldNameToParent).NewValue :=
            DeltaDS.DatasetField.DataSet.FieldByName(
              pk.ParentFieldName).AsVariant;
          DeltaDS.FieldByName(pk.FieldName).NewValue :=
            GetNext(FQuery, pk.TableName, pk.FieldName);
        end;
      ikInheritedTable:
        DeltaDS.FieldByName(pk.FieldName).NewValue :=
          DeltaDS.DatasetField.DataSet.FieldByName(
            pk.ParentFieldName).AsVariant;
    end;
  finally
    FQuery.SQLConnection := nil;
    FreeAndNil(FQuery);
    Unlock;
  end;
end;

function TDynamicSequence.GetNext(Query: TSQLQuery;
  const sTabela, sCampo: String): Integer;
begin
  with Query do
  begin
    try
      Close;
      SQL.Clear;
      SQL.Text := 'SELECT MAX(' + sCampo + ') FROM ' + sTabela;
      Open;
      if EOF then
        Result := 1
      else
        Result := Fields[0].AsInteger + 1;
    finally
      Close;
    end;
  end;
end;

initialization

finalization
  if Assigned(_DynamicSequence) then
    _DynamicSequence.Free;

end.
