unit Dbx4SQLite3;

interface

uses SysUtils, DBXCommon, ClassRegistry, SQLite3, DB, DBXPlatform, Math;

type 

  TDbx4SQLite3DriverLoader = class(TDBXDriverLoader)
  public
    function Load(DriverDef: TDBXDriverDef): TDBXDriver; override;
  end;

  TDbx4SQLite3Driver = class(TDBXDriver)
  protected
    function CreateConnection(ConnectionBuilder: TDBXConnectionBuilder): TDBXConnection; override;
    procedure Close; override;
  public
    function GetDriverVersion: WideString; override;
  end;

  TDbx4SQLite3Connection = class(TDBXConnection)
  private
    class var FDB: TSQLiteDB;
    FDatabaseMetaData: TDBXDatabaseMetaData;
  protected
    constructor Create(ConnectionBuilder: TDBXConnectionBuilder);
    procedure DerivedOpen(); override;
    procedure Close; override;
    function  DerivedCreateCommand: TDBXCommand; override;
    function  GetDatabaseMetaData: TDBXDatabaseMetaData; override;
  public
    class procedure TrataErro(Erro: Integer);
    destructor Destroy; override;
  end;

  TDbx4SQLite3DatabaseMetadata = class(TDBXDatabaseMetaData)
  private
    FSupportsTransactions: Boolean;
    FSupportsNestedTransactions: Boolean;
    FSupportsRowSetSize: Boolean;
    FMaxCommands: TInt32;
    FProcedureQuoteChar: WideString;
    FQuoteChar: WideString;
  published
  public
    constructor Create(DBXContext: TDBXContext);
    property QuoteChar: WideString read FQuoteChar;
    property ProcedureQuoteChar: WideString read FProcedureQuoteChar;
    property SupportsTransactions: Boolean read FSupportsTransactions;
    property SupportsNestedTransactions: Boolean read FSupportsNestedTransactions;
    property MaxCommands: TInt32 read FMaxCommands;
    property SupportsRowSetSize: Boolean read FSupportsRowSetSize;
  end;

  TDbx4SQLite3Command = class(TDBXCommand)
  private
    FSTMT: TSQLiteStmt;
  protected
    procedure DerivedOpen; override;
    procedure DerivedClose; override;
    function  DerivedExecuteQuery: TDBXReader; override;
    procedure DerivedPrepare; override;
  end;

  TDbx4SQLite3Reader = class(TDBXReader)
  private
    FSTMT: TSQLiteStmt;
    Empty: Boolean;
    PrimeiroNext: Boolean;
    function AchaLengthTipo(Tipo: String): Integer;
    procedure WhatIsColumnType(Ordinal: Integer; var Column: TDBXValueType);
  protected
    function  DerivedNext: Boolean; override;
  public
    constructor Create(DBXContext: TDBXContext; DbxRow: TDBXRow;
      ByteReader: TDBXByteReader; STMT: TSQLiteStmt);
  end;

  TDbx4SQLite3ByteReader = class(TDBXByteReader)
  private
    FReader: TDbx4SQLite3Reader;
    FSTMT: TSQLiteStmt;
  public
    constructor Create(DBXContext: TDBXContext; STMT: TSQLiteStmt);
    procedure SetReader(Reader: TDBXReader);
    procedure GetWideString(Ordinal: TInt32; const Value: TBytes; Offset: TInt32; var IsNull: LongBool); override;
    procedure GetInt32(Ordinal: TInt32; const Value: TBytes; Offset: TInt32; var IsNull: LongBool); override;
    procedure GetDouble(Ordinal: TInt32; const Value: TBytes; Offset: TInt32; var IsNull: LongBool); override;
    procedure GetByteLength(Ordinal: TInt32; var Length: Int64; var IsNull: LongBool); override;
    function  GetBytes(Ordinal: TInt32; Offset: Int64; const Value: TBytes;
                       ValueOffset, Length: Int64; var IsNull: LongBool): Int64; override;
    procedure GetInt64(Ordinal: TInt32; const Value: TBytes; Offset: TInt32; var IsNull: LongBool); virtual; abstract;
    procedure GetAnsiString(Ordinal: TInt32; const Value: TBytes; Offset: TInt32; var IsNull: LongBool); virtual; abstract;
    procedure GetInt16(Ordinal: TInt32; const Value: TBytes; Offset: TInt32; var IsNull: LongBool); virtual; abstract;
    procedure GetBcd(Ordinal: TInt32; const Value: TBytes; Offset: TInt32; var IsNull: LongBool); virtual; abstract;
    procedure GetTimeStamp(Ordinal: TInt32; const Value: TBytes; Offset: TInt32; var IsNull: LongBool); virtual; abstract;
    procedure GetTime(Ordinal: TInt32; const Value: TBytes; Offset: TInt32; var IsNull: LongBool); virtual; abstract;
    procedure GetDate(Ordinal: TInt32; const Value: TBytes; Offset: TInt32; var IsNull: LongBool); virtual; abstract;
  end;

implementation

{ TDbx4SQLite3DriverLoader }

function TDbx4SQLite3DriverLoader.Load(DriverDef: TDBXDriverDef): TDBXDriver;
begin
  if DriverDef.FDriverName = 'Dbx4SQLite3' then
    Result := TDbx4SQLite3Driver.Create
  else
    Result := nil;
end;

{ TDbx4SQLite3Driver }

procedure TDbx4SQLite3Driver.Close;
begin
  inherited;
end;

function TDbx4SQLite3Driver.CreateConnection(
  ConnectionBuilder: TDBXConnectionBuilder): TDBXConnection;
begin
  Result := TDbx4SQLite3Connection.Create(ConnectionBuilder);
end;

function TDbx4SQLite3Driver.GetDriverVersion: WideString;
begin
  Result := DBXVersion40;
end;

{ TDbx4SQLite3Connection }

procedure TDbx4SQLite3Connection.Close;
begin
  TrataErro(SQLite3_Close(FDB));
  inherited;
end;

constructor TDbx4SQLite3Connection.Create(ConnectionBuilder: TDBXConnectionBuilder);
begin
  inherited;
  FDatabaseMetaData := TDbx4SQLite3DatabaseMetadata.Create(FDBXContext);
end;

function TDbx4SQLite3Connection.DerivedCreateCommand: TDBXCommand;
begin
  Result := TDbx4SQLite3Command.Create(FDBXContext);
end;

procedure TDbx4SQLite3Connection.DerivedOpen;
var
  Database: String;
begin
  inherited;
  FDB := nil;
  Database := FConnectionProperties.Values[TDBXPropertyNames.Database];
  if SQLite3_Open(PAnsiChar(Database), FDB) <> SQLITE_OK then
    raise Exception.Create('Erro ao conectar ao banco de dados.');
end;

destructor TDbx4SQLite3Connection.Destroy;
begin
  FreeAndNil(FDatabaseMetaData);
  inherited;
end;

function TDbx4SQLite3Connection.GetDatabaseMetaData: TDBXDatabaseMetaData;
begin
  Result := FDatabaseMetaData;
end;

class procedure TDbx4SQLite3Connection.TrataErro(Erro: Integer);
begin
  if not Erro in [SQLITE_DONE, SQLITE_OK] then
    raise EDatabaseError.Create(SQLite3_ErrMsg(FDB));
end;

{ TDbx4SQLite3Command }

procedure TDbx4SQLite3Command.DerivedClose;
begin
  SQLite3_Finalize(FSTMT);
  inherited;
end;

function TDbx4SQLite3Command.DerivedExecuteQuery: TDBXReader;
var
  ReaderByte: TDbx4SQLite3ByteReader;
begin
  ReaderByte := TDbx4SQLite3ByteReader.Create(FDBXContext, FSTMT);
  Result := TDbx4SQLite3Reader.Create(FDBXContext, TDBXRow.Create(), ReaderByte, FSTMT);
  ReaderByte.SetReader(Result);
end;

procedure TDbx4SQLite3Command.DerivedOpen;
begin
  inherited;
  FSTMT := nil;
end;

procedure TDbx4SQLite3Command.DerivedPrepare;
var
  SQL: String;
  FIgnore: PChar;
begin
  SQL := Text;
  TDbx4SQLite3Connection.TrataErro(SQLite3_Prepare_v2(TDbx4SQLite3Connection.FDB, PChar(SQL), -1, FSTMT, FIgnore));
  inherited;
end;

{ TDbx4SQLite3ReaderDatabase }

constructor TDbx4SQLite3Reader.Create(DBXContext: TDBXContext; DbxRow: TDBXRow;
      ByteReader: TDBXByteReader; STMT: TSQLiteStmt);
var
  Ordinal:            TInt32;
  Column:             TDBXValueType;
  ColumnCount:        TInt32;
  Values:             TDBXValueArray;
begin
  inherited Create(DBXContext, TDBXRow.Create(), ByteReader);
  FSTMT := STMT;

  //Necessrio para obter o tipo dos dados retornados
  Empty := Sqlite3_Step(STMT) <> SQLITE_ROW;
  PrimeiroNext := True;

  ColumnCount := SQLite3_ColumnCount(FSTMT);
  SetLength(Values, ColumnCount);

  for Ordinal := 0 to High(Values) do
  begin
    Column                := TDBXDriverHelp.CreateTDBXValueType(DBXContext);
    WhatIsColumnType(Ordinal, Column);
    Column.Ordinal        := Ordinal;
    Column.ValueTypeFlags := 0;
    Values[Ordinal] := TDBXValue.CreateValue(FDBXContext, Column, FDbxRow, true);
  end;

  SetValues(Values);
end;

function TDbx4SQLite3Reader.DerivedNext: Boolean;
begin
  if PrimeiroNext then
  begin
    Result := not Empty;
    PrimeiroNext := False;
  end
  else
    Result := Sqlite3_Step(FSTMT) = SQLITE_ROW;
end;

function TDbx4SQLite3Reader.AchaLengthTipo(Tipo: String): Integer;
var
  PosP: Integer;
  Aux: String;
begin
  Result := 0;
  Aux := '';
  PosP := Pos('(', Tipo);
  if PosP > 0 then
  begin
    Inc(PosP);
    while (Tipo[PosP] in ['0'..'9']) do
    begin
      Aux := Aux + Tipo[PosP];
      Inc(PosP);
    end;
    TryStrToInt(Aux, Result);
  end;
end;

procedure TDbx4SQLite3Reader.WhatIsColumnType(Ordinal: Integer; var Column: TDBXValueType);
var
  TipoDecl: String;
  Tipo: Integer;
  Nome: PChar;
begin
  Nome := Sqlite3_ColumnName(FSTMT, Ordinal);
  Column.Name := Nome;
  Column.DataType := TDBXDataTypes.WideStringType;
  Column.SubType := 0;
  Column.Precision := 0;
  Column.Size := 0;
  Column.Scale := 0;

  Tipo := Sqlite3_ColumnType(FSTMT, Ordinal);
  case Tipo of
    SQLITE_INTEGER: TipoDecl := 'INT';
    SQLITE_FLOAT: TipoDecl := 'FLOAT';
    SQLITE_TEXT: TipoDecl := 'TEXT';
    SQLITE_BLOB: TipoDecl := 'BLOB';
  end;

  if Tipo in [0, 5] then
    TipoDecl := Sqlite3_ColumnDeclType(FSTMT, Ordinal);

  if Pos('INT', TipoDecl) > 0 then
  begin
    Column.DataType := TDBXDataTypes.Int32Type;
    Column.Size := SizeOf(Integer);
  end
  else
  if (TipoDecl = 'FLOAT') or (TipoDecl = 'REAL') then
  begin
    Column.DataType := TDBXDataTypes.DoubleType;
    Column.Size := SizeOf(Double);
  end
  else
  if (TipoDecl = 'BLOB') then
  begin
    Column.DataType := TDBXDataTypes.BlobType;
    Column.SubType := TDBXDataTypes.BinarySubType; 
  end
  else
  begin
    Column.Precision := AchaLengthTipo(TipoDecl);
    if Column.Precision > 0 then
      Column.Size := Column.Precision + 1
    else
    begin
      Column.DataType := TDBXDataTypes.BlobType;
      Column.SubType := TDBXDataTypes.WideMemoSubType;
    end
  end;
end;

{ TDbx4SQLite3DatabaseMetadata }

constructor TDbx4SQLite3DatabaseMetadata.Create(DBXContext: TDBXContext);
begin
  FQuoteChar := '';
  FProcedureQuoteChar := '';
  FMaxCommands := 0;
  FSupportsTransactions := True;
  FSupportsNestedTransactions := False;
  FSupportsRowSetSize := False;
end;

{ TDbx4SQLite3ByteReader }

constructor TDbx4SQLite3ByteReader.Create(DBXContext: TDBXContext; STMT: TSQLiteStmt);
begin
  inherited Create(DBXContext);
  FSTMT := STMT;
end;

procedure TDbx4SQLite3ByteReader.GetByteLength(Ordinal: TInt32;
  var Length: Int64; var IsNull: LongBool);
begin
  Length := Sqlite3_ColumnBytes(FSTMT, Ordinal)+1;
  IsNull := Length = 1;
  if FReader.ValueType[Ordinal].SubType = TDBXDataTypes.WideMemoSubType then
    Length := Length*SizeOf(WideChar);

end;

function TDbx4SQLite3ByteReader.GetBytes(Ordinal: TInt32; Offset: Int64;
  const Value: TBytes; ValueOffset, Length: Int64; var IsNull: LongBool): Int64;
var
  ValueAux: PChar;
  ValueWide: WideString;
begin
  IsNull := Length = 0;
  if not IsNull then
  begin
    ValueAux := Sqlite3_ColumnBlob(FSTMT, Ordinal);
    
    if FReader.ValueType[Ordinal].SubType = TDBXDataTypes.WideMemoSubType then
    begin
      ValueWide := ValueAux;
      Move(ValueWide[1], Value[0], SizeOf(WideChar)*Length);
    end
    else
      Move(ValueAux, Value[0], Length);
  end;
end;

procedure TDbx4SQLite3ByteReader.GetDouble(Ordinal: TInt32; const Value: TBytes;
  Offset: TInt32; var IsNull: LongBool);
begin
  inherited;
  IsNull := Sqlite3_ColumnText(FSTMT, Ordinal) = nil;
  if not IsNull then
    TDBXPlatform.CopyInt64(TDBXPlatform.DoubleToInt64Bits(Sqlite3_ColumnDouble(FSTMT, Ordinal)), Value, 0);
end;

procedure TDbx4SQLite3ByteReader.GetInt32(Ordinal: TInt32; const Value: TBytes;
  Offset: TInt32; var IsNull: LongBool);
begin
  IsNull := Sqlite3_ColumnText(FSTMT, Ordinal) = nil;
  if not IsNull then
    TDBXPlatform.CopyInt32(Sqlite3_ColumnInt(FSTMT, Ordinal), Value, Offset);
end;

procedure TDbx4SQLite3ByteReader.GetWideString(Ordinal: TInt32;
  const Value: TBytes; Offset: TInt32; var IsNull: LongBool);
var
  ValueAux: PAnsiChar;
  ValueWide: WideString;
begin
  ValueAux := Sqlite3_ColumnText(FSTMT, Ordinal);
  IsNull := ValueAux = nil;
  if not IsNull then
  begin
    ValueWide := UTF8Decode(ValueAux);
    Move(ValueWide[1], Value[0], Min(Length(Value), SizeOf(WideString)*Length(ValueWide)));
  end;
end;

procedure TDbx4SQLite3ByteReader.SetReader(Reader: TDBXReader);
begin
  FReader := TDbx4SQLite3Reader(Reader);
end;

initialization
  TClassRegistry.GetClassRegistry.RegisterClass(TDbx4SQLite3DriverLoader.ClassName, TDbx4SQLite3DriverLoader);

finalization
  TClassRegistry.GetClassRegistry.UnregisterClass(TDbx4SQLite3DriverLoader.ClassName);

end.
