unit Dbx4SQLite3;

interface

uses SysUtils, DBXCommon, ClassRegistry, SQLite3, DBXPlatform, Math;

type

  TDbx4SQLite3DriverLoader = class(TDBXDriverLoader)
  public
    function Load(DriverDef: TDBXDriverDef): TDBXDriver; override;
  end;

  TDbx4SQLite3Driver = class(TDBXDriver)
  protected
    function CreateConnection(ConnectionBuilder: TDBXConnectionBuilder): TDBXConnection; override;
    procedure Close; override;
  public
    function GetDriverVersion: WideString; override;
  end;

  TDbx4SQLite3DatabaseMetadata = class
  private
    FQuoteChar: WideString;
    FProcedureQuoteChar: WideString;
    FSupportsTransactions: Boolean;
    FSupportsNestedTransactions: Boolean;
    FMaxCommands: TInt32;
    FSupportsRowSetSize: Boolean;
  end;

  TDbx4SQLite3Connection = class(TDBXConnection)
  private
    class var FDB: TSQLiteDB;
    FDatabaseMetaData: TDBXDatabaseMetaData;
  protected
    procedure DerivedOpen(); override;
    procedure Close; override;
    function  DerivedCreateCommand: TDBXCommand; override;
    function  CreateAndBeginTransaction(Isolation: TDBXIsolation): TDBXTransaction; override;
    procedure Commit(Transaction: TDBXTransaction); override;
    procedure Rollback(Transaction: TDBXTransaction); override;
  public
    class procedure TrataErro(Erro: Integer);
    destructor Destroy; override;
    function GetDatabaseMetaData: TDBXDatabaseMetaData; override;
    constructor Create(ConnectionBuilder: TDBXConnectionBuilder);
  end;

  TDbx4SQLite3Command = class(TDBXCommand)
  private
    FSTMT: TSQLiteStmt;
  protected
    procedure DerivedOpen; override;
    procedure DerivedClose; override;
    function DerivedExecuteQuery: TDBXReader; override;
    procedure DerivedPrepare; override;
    function CreateParameterRow: TDBXRow; override;
    function GetRowsAffected: Int64; override;
  end;

  TDbx4SQLite3Reader = class(TDBXReader)
  private
    FSTMT: TSQLiteStmt;
    Empty: Boolean;
    PrimeiroNext: Boolean;
    function AchaLengthTipo(Tipo: String): Integer;
    procedure WhatIsColumnType(Ordinal: Integer; var Column: TDBXValueType);
    function IsNullable(Ordinal: Integer; Nome: string): Boolean;
  protected
    procedure DerivedClose; override;
    function DerivedNext: Boolean; override;
  public
    constructor Create(DBXContext: TDBXContext; DbxRow: TDBXRow;
      ByteReader: TDBXByteReader; STMT: TSQLiteStmt);
  end;

  TDbx4SQLite3ByteReader = class(TDBXByteReader)
  private
    FReader: TDbx4SQLite3Reader;
    FSTMT: TSQLiteStmt;
  public
    constructor Create(DBXContext: TDBXContext; STMT: TSQLiteStmt);
    procedure SetReader(Reader: TDBXReader);
    procedure GetWideString(Ordinal: TInt32; const Value: TBytes; Offset: TInt32; var IsNull: LongBool); override;
    procedure GetInt32(Ordinal: TInt32; const Value: TBytes; Offset: TInt32; var IsNull: LongBool); override;
    procedure GetDouble(Ordinal: TInt32; const Value: TBytes; Offset: TInt32; var IsNull: LongBool); override;
    procedure GetByteLength(Ordinal: TInt32; var Length: Int64; var IsNull: LongBool); override;
    function GetBytes(Ordinal: TInt32; Offset: Int64; const Value: TBytes;
      ValueOffset, VLength: Int64; var IsNull: LongBool): Int64; override;
  end;

  TDbx4SQLite3Row = class(TDBXRow)
  private
    FSTMT: TSQLiteStmt;
  protected
    procedure SetValueType(ValueType: TDBXValueType); override;
    procedure SetNull(DbxValue: TDBXValue); override;
    procedure SetInt32(DbxValue: TDBXInt32Value; Value: TInt32); override;
    procedure SetDouble(DbxValue: TDBXDoubleValue; Value: Double); override;
    procedure SetWideString(DbxValue: TDBXWideStringValue; const Value: WideString); override;
    procedure SetDynamicBytes(DbxValue: TDBXValue; Offset: Int64; const Buffer: TBytes; BufferOffset: Int64; VLength: Int64); override;
  public
    constructor Create(DBXContext: TDBXContext; STMT: TSQLiteStmt);
  end;
  
implementation

{ TDbx4SQLite3DriverLoader }

function TDbx4SQLite3DriverLoader.Load(DriverDef: TDBXDriverDef): TDBXDriver;
begin
  if DriverDef.FDriverName = 'Dbx4SQLite3' then
    Result := TDbx4SQLite3Driver.Create
  else
    Result := nil;
end;

{ TDbx4SQLite3Driver }

procedure TDbx4SQLite3Driver.Close;
begin
  inherited;
end;

function TDbx4SQLite3Driver.CreateConnection(
  ConnectionBuilder: TDBXConnectionBuilder): TDBXConnection;
begin
  Result := TDbx4SQLite3Connection.Create(ConnectionBuilder);
end;

function TDbx4SQLite3Driver.GetDriverVersion: WideString;
begin
  Result := DBXVersion40;
end;

{ TDbx4SQLite3Connection }

procedure TDbx4SQLite3Connection.Close;
begin
  inherited;
  TrataErro(SQLite3_Close(FDB));
end;

constructor TDbx4SQLite3Connection.Create(
  ConnectionBuilder: TDBXConnectionBuilder);
begin
  inherited;
  FDatabaseMetaData := TDBXDatabaseMetaData.Create;
  TDbx4SQLite3DatabaseMetadata(FDatabaseMetaData).FSupportsTransactions := True;
end;

function TDbx4SQLite3Connection.CreateAndBeginTransaction(
  Isolation: TDBXIsolation): TDBXTransaction;
var
  FIgnore: PChar;
begin
  inherited;
  TrataErro(SQLite3_Exec(FDB, 'BEGIN TRANSACTION', nil, nil, FIgnore));
  Result := TDBXTransaction.Create;
end;


procedure TDbx4SQLite3Connection.Commit(Transaction: TDBXTransaction);
var
  FIgnore: PChar;
begin
  inherited;
  TrataErro(SQLite3_Exec(FDB, 'COMMIT TRANSACTION', nil, nil, FIgnore));
end;

procedure TDbx4SQLite3Connection.Rollback(Transaction: TDBXTransaction);
var
  FIgnore: PChar;
begin
  inherited;
  TrataErro(SQLite3_Exec(FDB, 'ROLLBACK TRANSACTION', nil, nil, FIgnore));
end;

function TDbx4SQLite3Connection.DerivedCreateCommand: TDBXCommand;
begin
  Result := TDbx4SQLite3Command.Create(FDBXContext);
end;

procedure TDbx4SQLite3Connection.DerivedOpen;
var
  Database: String;
begin
  inherited;
  FDB := nil;
  Database := FConnectionProperties.Values[TDBXPropertyNames.Database];
  if SQLite3_Open(PAnsiChar(Database), FDB) <> SQLITE_OK then
    raise Exception.Create('Erro ao conectar ao banco de dados.');
end;

destructor TDbx4SQLite3Connection.Destroy;
begin
  FreeAndNil(FDatabaseMetaData);
  inherited;
end;

function TDbx4SQLite3Connection.GetDatabaseMetaData: TDBXDatabaseMetaData;
begin
  Result := FDatabaseMetaData;
end;

class procedure TDbx4SQLite3Connection.TrataErro(Erro: Integer);
begin
  if not (Erro in [SQLITE_DONE, SQLITE_OK]) then
    raise Exception.Create(SQLite3_ErrMsg(FDB));
end;

{ TDbx4SQLite3Command }

function TDbx4SQLite3Command.CreateParameterRow: TDBXRow;
begin
  Result := TDbx4SQLite3Row.Create(FDBXContext, FSTMT);
end;

procedure TDbx4SQLite3Command.DerivedClose;
begin
  SQLite3_Finalize(FSTMT);
  inherited;
end;

function TDbx4SQLite3Command.DerivedExecuteQuery: TDBXReader;
var
  ReaderByte: TDbx4SQLite3ByteReader;
begin
  ReaderByte := TDbx4SQLite3ByteReader.Create(FDBXContext, FSTMT);
  Result := TDbx4SQLite3Reader.Create(FDBXContext, TDBXRow.Create(), ReaderByte, FSTMT);
  ReaderByte.SetReader(Result);
end;

procedure TDbx4SQLite3Command.DerivedOpen;
begin
  inherited;
  FSTMT := nil;
end;

procedure TDbx4SQLite3Command.DerivedPrepare;
var
  SQL: String;
  FIgnore: PChar;
begin
  SQL := Text;
  TDbx4SQLite3Connection.TrataErro(SQLite3_Prepare_v2(TDbx4SQLite3Connection.FDB, PChar(SQL), -1, FSTMT, FIgnore));
  inherited;
end;

function TDbx4SQLite3Command.GetRowsAffected: Int64;
begin
  Result := SQLite3_Changes(TDbx4SQLite3Connection.FDB);
end;

{ TDbx4SQLite3Reader }

constructor TDbx4SQLite3Reader.Create(DBXContext: TDBXContext; DbxRow: TDBXRow;
  ByteReader: TDBXByteReader; STMT: TSQLiteStmt);
var
  Ordinal: TInt32;
  Column: TDBXValueType;
  ColumnCount: TInt32;
  Values: TDBXValueArray;
begin
  inherited Create(DBXContext, TDBXRow.Create(), ByteReader);
  FSTMT := STMT;

  //Necessrio para obter o tipo dos dados retornados
  Empty := Sqlite3_Step(STMT) <> SQLITE_ROW;
  PrimeiroNext := True;
  ColumnCount := SQLite3_ColumnCount(FSTMT);
  SetLength(Values, ColumnCount);

  for Ordinal := 0 to High(Values) do
  begin
    Column := TDBXDriverHelp.CreateTDBXValueType(DBXContext);
    WhatIsColumnType(Ordinal, Column);
    Column.Ordinal := Ordinal;
    Column.ValueTypeFlags := 0;
    Values[Ordinal] := TDBXValue.CreateValue(FDBXContext, Column, FDbxRow, true);
  end;
  SetValues(Values);
end;

procedure TDbx4SQLite3Reader.DerivedClose;
begin
  inherited;
end;

function TDbx4SQLite3Reader.DerivedNext: Boolean;
begin
  if PrimeiroNext then
  begin
    Result := not Empty;
    PrimeiroNext := False;
  end
  else
    Result := Sqlite3_Step(FSTMT) = SQLITE_ROW;
end;

function TDbx4SQLite3Reader.IsNullable(Ordinal: Integer; Nome: string): Boolean;
var
  Table: PChar;
  PragmaSTMT: TSQLiteStmt;
  FIgnore: PChar;
  SQL: string;
begin
  Table := Sqlite3_ColumnTableName(FSTMT, Ordinal);
  if (Table <> nil) then
  begin
    SQL := Format('pragma table_info(%s)', [Table]);
    TDbx4SQLite3Connection.TrataErro(
      SQLite3_Prepare_v2(TDbx4SQLite3Connection.FDB, PChar(SQL), -1, PragmaSTMT, FIgnore));
    try
      while (Sqlite3_Step(PragmaSTMT) = SQLITE_ROW) do
      begin
        if (Sqlite3_ColumnText(PragmaSTMT, 1) = Nome) then
        begin
          Result := Sqlite3_ColumnInt(PragmaSTMT, 3) = 0;
          Exit;
        end;
      end;
    finally
      TDbx4SQLite3Connection.TrataErro(
        SQLite3_Finalize(PragmaSTMT));
    end;
  end;

  Result := True;
end;

function TDbx4SQLite3Reader.AchaLengthTipo(Tipo: String): Integer;
var
  PosP: Integer;
  Aux: String;
begin
  Result := 0;
  Aux := '';
  PosP := Pos('(', Tipo);
  if PosP > 0 then
  begin
    Inc(PosP);
    while (Tipo[PosP] in ['0'..'9']) do
    begin
      Aux := Aux + Tipo[PosP];
      Inc(PosP);
    end;
    TryStrToInt(Aux, Result);
  end;
end;

procedure TDbx4SQLite3Reader.WhatIsColumnType(Ordinal: Integer; var Column: TDBXValueType);
var
  TipoDecl: String;
  Tipo: Integer;
  Nome: PChar;
begin
  Nome := Sqlite3_ColumnName(FSTMT, Ordinal);
  Column.Name := Nome;
  Column.DataType := TDBXDataTypes.WideStringType;
  Column.SubType := 0;
  Column.Precision := 0;
  Column.Size := 0;
  Column.Scale := 0;
  Column.Nullable := IsNullable(Ordinal, Nome);

  Tipo := Sqlite3_ColumnType(FSTMT, Ordinal);
  case Tipo of
    SQLITE_INTEGER: TipoDecl := 'INT';
    SQLITE_FLOAT: TipoDecl := 'FLOAT';
    SQLITE_TEXT: TipoDecl := 'TEXT';
    SQLITE_BLOB: TipoDecl := 'BLOB';
  end;

  if Tipo in [0, 5] then
    TipoDecl := Sqlite3_ColumnDeclType(FSTMT, Ordinal);

  if Pos('INT', TipoDecl) > 0 then
  begin
    Column.DataType := TDBXDataTypes.Int32Type;
    Column.Size := SizeOf(Integer);
  end
  else
  if (TipoDecl = 'FLOAT') or (TipoDecl = 'REAL') then
  begin
    Column.DataType := TDBXDataTypes.DoubleType;
    Column.Size := SizeOf(Double);
  end
  else
  if (TipoDecl = 'BLOB') then
  begin
    Column.DataType := TDBXDataTypes.BlobType;
    Column.SubType := TDBXDataTypes.BinarySubType;
  end
  else
  begin
    Column.Precision := AchaLengthTipo(TipoDecl);
    if Column.Precision > 0 then
      Column.Size := Column.Precision + 1
    else
    begin
      Column.DataType := TDBXDataTypes.BlobType;
      Column.SubType := TDBXDataTypes.WideMemoSubType;
    end
  end;
end;

{ TDbx4SQLite3ByteReader }

constructor TDbx4SQLite3ByteReader.Create(DBXContext: TDBXContext; STMT: TSQLiteStmt);
begin
  inherited Create(DBXContext);
  FSTMT := STMT;
end;

procedure TDbx4SQLite3ByteReader.GetByteLength(Ordinal: TInt32;
  var Length: Int64; var IsNull: LongBool);
begin
  Length := Sqlite3_ColumnBytes(FSTMT, Ordinal);
  IsNull := Length = 0;
  if FReader.ValueType[Ordinal].SubType = TDBXDataTypes.WideMemoSubType then
    Length := Length*SizeOf(WideChar);
end;

function TDbx4SQLite3ByteReader.GetBytes(Ordinal: TInt32; Offset: Int64;
  const Value: TBytes; ValueOffset, VLength: Int64; var IsNull: LongBool): Int64;
var
  ValueAux: PChar;
  ValueWide: WideString;
begin
  IsNull := VLength = 0;
  if not IsNull then
  begin
    ValueAux := Sqlite3_ColumnBlob(FSTMT, Ordinal);
    if FReader.ValueType[Ordinal].SubType = TDBXDataTypes.WideMemoSubType then
    begin
      ValueWide := UTF8Decode(ValueAux);
      Move(ValueWide[1], Value[0], SizeOf(WideChar)*Length(ValueWide));
    end
    else
      Move(ValueAux, Value[0], VLength);
  end;
end;

procedure TDbx4SQLite3ByteReader.GetDouble(Ordinal: TInt32; const Value: TBytes;
  Offset: TInt32; var IsNull: LongBool);
begin
  inherited;
  IsNull := Sqlite3_ColumnText(FSTMT, Ordinal) = nil;
  if not IsNull then
  TDBXPlatform.CopyInt64(TDBXPlatform.DoubleToInt64Bits(Sqlite3_ColumnDouble(FSTMT, Ordinal)), Value, 0);
end;

procedure TDbx4SQLite3ByteReader.GetInt32(Ordinal: TInt32; const Value: TBytes;
Offset: TInt32; var IsNull: LongBool);
begin
  IsNull := Sqlite3_ColumnText(FSTMT, Ordinal) = nil;
  if not IsNull then
  TDBXPlatform.CopyInt32(Sqlite3_ColumnInt(FSTMT, Ordinal), Value, Offset);
end;

procedure TDbx4SQLite3ByteReader.GetWideString(Ordinal: TInt32;
const Value: TBytes; Offset: TInt32; var IsNull: LongBool);
var
  ValueAux: PAnsiChar;
  ValueWide: WideString;
begin
  ValueAux := Sqlite3_ColumnText(FSTMT, Ordinal);
  IsNull := ValueAux = nil;
  if not IsNull then
  begin
    ValueWide := UTF8Decode(ValueAux);
    Move(ValueWide[1], Value[0], Min(Length(Value), SizeOf(WideChar)*Length(ValueWide)));
  end;
end;

procedure TDbx4SQLite3ByteReader.SetReader(Reader: TDBXReader);
begin
  FReader := TDbx4SQLite3Reader(Reader);
end;

{ TDbx4SQLite4Row }

constructor TDbx4SQLite3Row.Create(DBXContext: TDBXContext; STMT: TSQLiteStmt);
begin
  inherited Create(DBXContext);
  FSTMT := STMT;
end;

procedure TDbx4SQLite3Row.SetValueType(ValueType: TDBXValueType);
begin
  //inherited;
end;

procedure TDbx4SQLite3Row.SetDouble(DbxValue: TDBXDoubleValue; Value: Double);
begin
  TDbx4SQLite3Connection.TrataErro(
    SQLite3_Bind_Double(FSTMT, DbxValue.ValueType.Ordinal+1, Value));
end;

procedure TDbx4SQLite3Row.SetDynamicBytes(DbxValue: TDBXValue; Offset: Int64;
  const Buffer: TBytes; BufferOffset, VLength: Int64);
var
  Aux: string;
begin
  if DbxValue.ValueType.SubType = TDBXDataTypes.WideMemoSubType then
  begin
    Aux := UTF8Encode(TDBXPlatform.BytesToWideStr(Buffer));
    TDbx4SQLite3Connection.TrataErro(
      SQLite3_Bind_text(FSTMT, DbxValue.ValueType.Ordinal+1, PChar(Aux), Length(Aux), nil));
  end
  else
    TDbx4SQLite3Connection.TrataErro(
      SQLite3_Bind_Blob(FSTMT, DbxValue.ValueType.Ordinal+1, @Buffer[0], VLength, nil));
end;

procedure TDbx4SQLite3Row.SetInt32(DbxValue: TDBXInt32Value; Value: TInt32);
begin
  TDbx4SQLite3Connection.TrataErro(
    SQLite3_BindInt(FSTMT, DbxValue.ValueType.Ordinal+1, Value));
end;

procedure TDbx4SQLite3Row.SetNull(DbxValue: TDBXValue);
begin
  TDbx4SQLite3Connection.TrataErro(
    SQLite3_Bind_null(FSTMT, DbxValue.ValueType.Ordinal+1));
end;

procedure TDbx4SQLite3Row.SetWideString(DbxValue: TDBXWideStringValue;
  const Value: WideString);
var
  Aux: string;
begin
  Aux := UTF8Encode(Value);
  TDbx4SQLite3Connection.TrataErro(
    SQLite3_Bind_text(FSTMT, DbxValue.ValueType.Ordinal+1, PChar(Aux), Length(Aux), nil));
end;

initialization
  TClassRegistry.GetClassRegistry.RegisterClass(TDbx4SQLite3DriverLoader.ClassName, TDbx4SQLite3DriverLoader);

finalization
  TClassRegistry.GetClassRegistry.UnregisterClass(TDbx4SQLite3DriverLoader.ClassName);

end.
