unit Dbx4SQLite3Metadata;

interface

uses DBXMetaDataReader, DBXMetaDataCommandFactory, DBXTableStorage,
  DBXPlatformUtil, DBXSqlScanner, DBXMetaDataNames, DBXTypedTableStorage,
  SysUtils;

type

  TDbx4SQLite3MetaDataReader = class(TDBXBaseMetaDataReader)
  public
    function FetchSchemas(const Catalog: WideString): TDBXTableStorage; override;
    function FetchTables(const Catalog: WideString; const Schema: WideString;
      const TableName: WideString; const TableType: WideString): TDBXTableStorage; override;
    function FetchProcedures(const Catalog: WideString; const Schema: WideString;
      const ProcedureName: WideString; const ProcedureType: WideString): TDBXTableStorage; override;
    function FetchColumns(const Catalog: WideString; const Schema: WideString;
      const Table: WideString): TDBXTableStorage; override;
    function FetchIndexes(const Catalog: WideString; const Schema: WideString;
      const Table: WideString): TDBXTableStorage; override;
    function FetchIndexColumns(const Catalog: WideString;
      const Schema: WideString; const Table: WideString;
      const Index: WideString): TDBXTableStorage; override;
    function FetchForeignKeys(const Catalog: WideString; const Schema: WideString;
      const Table: WideString): TDBXTableStorage; override;
    function GetProductName: WideString; override;
    function FetchUsers: TDBXTableStorage; override;
  end;

  TDbx4SQLite3MapColumnStorage = class(TDBXTableStorage)
  private
    FOrigem: TDBXTableStorage;
    FColumns: TDBXColumnDescriptorArray;
  protected
    function GetCommand: TObject; override;
    function GetColumns: TDBXColumnDescriptorArray; override;
  public
    constructor Create(Origin: TDBXTableStorage);
    destructor Destroy; override;
    function GetString(Ordinal: Integer): WideString; override;
    function IsNull(Ordinal: Integer): Boolean; override;
    function Next: Boolean; override;
    procedure Close; override;
  end;

  TDbx4SQLite3MetadataCommandFactory = class(TDBXMetaDataCommandFactory)
  public
    function CreateMetaDataReader: TDBXMetaDataReader; override;
  end;

implementation

{ TDbx4SQLite3MetaDataReader }

function TDbx4SQLite3MetaDataReader.FetchColumns(const Catalog, Schema,
  Table: WideString): TDBXTableStorage;
var
  CursorAux: TDBXTableStorage;
  ParameterNames: TDBXWideStringArray;
  ParameterValues: TDBXWideStringArray;
  SQL: String;
begin
  SQL := Format('pragma table_info(''%s'')', [Table]);
  CursorAux := FContext.ExecuteQuery(SQL, ParameterNames, ParameterValues);
  Result := TDbx4SQLite3MapColumnStorage.Create(CursorAux);
end;

function TDbx4SQLite3MetaDataReader.FetchForeignKeys(const Catalog, Schema,
  Table: WideString): TDBXTableStorage;
begin
  Result := TDBXBaseMetaDataReader.TDBXEmptyTableCursor.Create(
    TDBXMetaDataCollectionIndex.ForeignKeys, TDBXMetaDataCollectionName.ForeignKeys,
    TDBXMetaDataCollectionColumns.CreateForeignKeysColumns);
end;

function TDbx4SQLite3MetaDataReader.FetchIndexColumns(const Catalog, Schema,
  Table, Index: WideString): TDBXTableStorage;
begin
  Result := TDBXBaseMetaDataReader.TDBXEmptyTableCursor.Create(
    TDBXMetaDataCollectionIndex.IndexColumns, TDBXMetaDataCollectionName.IndexColumns,
    TDBXMetaDataCollectionColumns.CreateIndexesColumns);
end;

function TDbx4SQLite3MetaDataReader.FetchIndexes(const Catalog, Schema,
  Table: WideString): TDBXTableStorage;
const
  SQL = 'select null, null, null, name from sqlite_master where type = ''index'' and tbl_name = :table';
var
  ParameterNames: TDBXWideStringArray;
  ParameterValues: TDBXWideStringArray;
  Cursor: TDBXTableStorage;
  Columns: TDBXColumnDescriptorArray;
begin
  SetLength(ParameterNames,1);
  ParameterNames[0] := 'table';
  SetLength(ParameterValues,1);
  ParameterValues[0] := Table;
  Cursor := FContext.ExecuteQuery(SQL, ParameterNames, ParameterValues);
  Columns := TDBXMetaDataCollectionColumns.CreateIndexesColumns;
  Result := TDBXBaseMetaDataReader.TDBXSanitizedTableCursor.Create(FContext, TDBXMetaDataCollectionIndex.Indexes, TDBXMetaDataCollectionName.Indexes, Columns, Cursor);
end;

function TDbx4SQLite3MetaDataReader.FetchProcedures(const Catalog, Schema,
  ProcedureName, ProcedureType: WideString): TDBXTableStorage;
begin
  Result := TDBXBaseMetaDataReader.TDBXEmptyTableCursor.Create(TDBXMetaDataCollectionIndex.Procedures,
    TDBXMetaDataCollectionName.Procedures, TDBXMetaDataCollectionColumns.CreateProceduresColumns);
end;

function TDbx4SQLite3MetaDataReader.FetchSchemas(
  const Catalog: WideString): TDBXTableStorage;
begin
  Result := TDBXBaseMetaDataReader.TDBXEmptyTableCursor.Create(TDBXMetaDataCollectionIndex.Schemas,
    TDBXMetaDataCollectionName.Schemas, TDBXMetaDataCollectionColumns.CreateSchemasColumns);
end;

function TDbx4SQLite3MetaDataReader.FetchTables(const Catalog, Schema,
  TableName, TableType: WideString): TDBXTableStorage;
const
  SQL = 'select null, null, name, type from sqlite_master where upper(type) in (%s, %s)';
var
  ParameterNames: TDBXWideStringArray;
  ParameterValues: TDBXWideStringArray;
  Cursor: TDBXTableStorage;
  Columns: TDBXColumnDescriptorArray;
  Tables, Views: String;
  TypeMask: Integer;
begin
  SetLength(ParameterNames, 0);
  SetLength(ParameterValues, 0);

  TypeMask := TDBXTableTypeParser.ParseTableTypes(TableType);
  Tables := QuotedStr(MakeTableTypeString(TDBXTableTypeFlag.Table, TypeMask));
  Views := QuotedStr(MakeTableTypeString(TDBXTableTypeFlag.View, TypeMask));

  Cursor := FContext.ExecuteQuery(Format(SQL, [Tables, Views]), ParameterNames, ParameterValues);
  Columns := TDBXMetaDataCollectionColumns.CreateTablesColumns;
  Result := TDBXBaseMetaDataReader.TDBXSanitizedTableCursor.Create(FContext,
    TDBXMetaDataCollectionIndex.Tables, TDBXMetaDataCollectionName.Tables,
    Columns, Cursor);
end;

function TDbx4SQLite3MetaDataReader.FetchUsers: TDBXTableStorage;
begin
  Result := TDBXBaseMetaDataReader.TDBXEmptyTableCursor.Create(TDBXMetaDataCollectionIndex.Users,
    TDBXMetaDataCollectionName.Users, TDBXMetaDataCollectionColumns.CreateUsersColumns);
end;

function TDbx4SQLite3MetaDataReader.GetProductName: WideString;
begin
  Result := 'Dbx4SQLite3';
end;

{ TDbx4SQLite3MetadaCommandFactor }

function TDbx4SQLite3MetadataCommandFactory.CreateMetaDataReader: TDBXMetaDataReader;
begin
  Result := TDbx4SQLite3MetaDataReader.Create;
end;

{ TDbx4SQLite3MapColumnStorage }

procedure TDbx4SQLite3MapColumnStorage.Close;
begin
  inherited;
  FOrigem.Close;
end;

constructor TDbx4SQLite3MapColumnStorage.Create(Origin: TDBXTableStorage);
begin
  inherited Create;
  FOrigem := Origin;
  FColumns := TDBXMetaDataCollectionColumns.CreateColumnsColumns;
end;

destructor TDbx4SQLite3MapColumnStorage.Destroy;
begin
  FreeAndNil(FOrigem);
  inherited;
end;

function TDbx4SQLite3MapColumnStorage.GetString(Ordinal: Integer): WideString;
begin
  Result := '';
  if Ordinal = 3 then //Column name
    Result := FOrigem.GetAsString(1);
end;

function TDbx4SQLite3MapColumnStorage.Next: Boolean;
begin
  Result := FOrigem.Next;
end;

function TDbx4SQLite3MapColumnStorage.GetColumns: TDBXColumnDescriptorArray;
begin
 Result := FColumns;
end;

function TDbx4SQLite3MapColumnStorage.GetCommand: TObject;
begin
  Result := FOrigem.Command; 
end;

function TDbx4SQLite3MapColumnStorage.IsNull(Ordinal: Integer): Boolean;
begin
  Result := True;
  if Ordinal = 3 then //Column name
    Result := FOrigem.IsNull(1);
end;

end.


