unit uOBJTextConverts;

interface
uses
  SysUtils, Classes, Graphics;

const
  BytesPerLine = 32; //Informa o nmero de caracteres por linha
  STR_INVALID_BINARY = 'Valor binrio invlido';

type
  EInvalidBinary = class(Exception); //Exceo personalizada para o formato da imagem

  TStreamConverter = class //Classe que converte valores binrios em string e vice-versa
  public
    class procedure HexTextToStream(const HexText: string; Stream: TStream);
    class function StreamToHexText(Stream: TStream): string;
  end;

  TImageConverter = class // Classe que converte imagens em texto e vice-versa
  public
    class procedure TextToImage(const TextImage: string; Image: TBitMap);
    class function ImageToText(Image: TBitmap): string;
  end;

implementation

class function TImageConverter.ImageToText(Image: TBitmap): string;
var
  MemStream: TMemoryStream;
begin
  Result := '';

  if Image.Empty then
    Exit;

  MemStream := TMemoryStream.Create;
  try
    Image.SaveToStream(MemStream); //Primeiro a imagem deve ser convertida para um Stream em memria.
    Result := TStreamConverter.StreamToHexText(MemStream); // Depois a classe TStreamConverter entra em ao para converter o Stream da imagem para texto
  finally
    MemStream.Free;
  end;
end;

class procedure TImageConverter.TextToImage(const TextImage: string;
  Image: TBitMap);
var
  MemStream: TStream;
begin
  MemStream := TMemoryStream.Create;

  try
    TStreamConverter.HexTextToStream(TextImage, MemStream); // Transforma o texto passado por parmetro em Stream
    Image.LoadFromStream(MemStream); // Em seguida transforma o Stream para imagem
  finally
    MemStream.Free;
  end;
end;

{ TStreamConverter }

class procedure TStreamConverter.HexTextToStream(
  const HexText: string; Stream: TStream);
var
  Count: Integer;
  Buffer: array[0..255] of Char;
  SourcePtr: PChar;
begin
  {
   A lgica deste mtodo foi baseada no mtodo HexToBinary da classe TParser da
   unit Classes.
  }

  SourcePtr := PChar(HexText);
  if SourcePtr^ <> '{' then // Faz a validao na string para garantir que  um texto vlido
    raise EInvalidBinary.Create(STR_INVALID_BINARY);

  while SourcePtr^ <> '}' do // Percorre a "lista" de ponteiros da string
  begin
    while not (SourcePtr^ in ['0'..'9', 'A'..'F']) do //Ignora os espaoes em branco ou diferentes de Hexa
        Inc(SourcePtr);
    Count := HexToBin(SourcePtr, Buffer, SizeOf(Buffer)); //Esta funo do Delphi converte um valor string para binrio
    if Count = 0 then // Se houve algum erro  porque o texto do stream no  um binrio vlido.
      raise EInvalidBinary.Create(STR_INVALID_BINARY);

    Stream.Write(Buffer, Count); //Armazena no stream os valores convertidos para binrio.
    Inc(SourcePtr, Count * 2)
  end;

  Stream.Seek(0, soFromBeginning); //Posiciona o ponteiro do Stream no incio.
end;

class function TStreamConverter.StreamToHexText(
  Stream: TStream): string;
var
  MultiLine: Boolean;
  I: Integer;
  Count: Longint;
  Buffer: array[0..BytesPerLine - 1] of Char;
  Text: array[0..BytesPerLine * 2 - 1] of Char;
  Reader: TReader;
  Writer: TWriter;
  StrStream: TStringStream;
  s: string;

  procedure WriteStr(const S: string); // Funo para escrita com ponteiro de string.
  begin
    Writer.Write(S[1], Length(S));
  end;

begin
  Result := '';
  Stream.Seek(0, soFromBeginning); //Posiciona o stream

  StrStream := TStringStream.Create(s); // Cria o Stream para o texto de sada

  Reader := TReader.Create(Stream, Stream.Size); // A classe TReader  utilizada para "navegar" em um Stream.
  Writer := TWriter.Create(StrStream, Stream.Size); // A classe TWriter  utilizada para gravar valores em um stream.
                                                    // Neste caso, cria o gravador para texto.
  try
    WriteStr('{');
    Count := Stream.Size;
    MultiLine := Count >= BytesPerLine;
    while Count > 0 do // Percorre o Stream para fazer a converso
    begin
      if MultiLine then WriteStr(#13#10);
      if Count >= 32 then I := 32 else I := Count;
      Reader.Read(Buffer, I); // Faz a leitura do stream
      BinToHex(Buffer, Text, I); // Converte os valores lidos do stream, de binrio para texto-hexa
      Writer.Write(Text, I * 2); //Escreve o valor convertido para texto-hexa no stream
      Dec(Count, I);
    end;
    WriteStr('}');

  finally
    Writer.Free;
    Reader.Free;

    Result := StrStream.DataString; // Retorna o valor convertido de binrio para texto-hexa
    StrStream.Free;
  end;

end;

end.

